/*
 * Copyright (c) 2015, 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 2.0, as published by the
 * Free Software Foundation.
 *
 * This program is also distributed with certain software (including but not
 * limited to OpenSSL) that is licensed under separate terms, as designated in a
 * particular file or component or in included license documentation. The
 * authors of MySQL hereby grant you an additional permission to link the
 * program and your derivative works with the separately licensed software that
 * they have included with MySQL.
 *
 * Without limiting anything contained in the foregoing, this file, which is
 * part of MySQL Connector/J, is also subject to the Universal FOSS Exception,
 * version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License, version 2.0,
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

package com.mysql.cj.xdevapi;

import java.util.Properties;

import com.mysql.cj.api.xdevapi.Session;
import com.mysql.cj.core.conf.url.ConnectionUrl;
import com.mysql.cj.core.conf.url.HostInfo;
import com.mysql.cj.core.exceptions.CJCommunicationsException;
import com.mysql.cj.core.exceptions.ExceptionFactory;
import com.mysql.cj.core.exceptions.InvalidConnectionAttributeException;

/**
 * SessionFactory is used for creation of sessions.
 * 
 * <pre>
 * SessionFactory xFactory = new SessionFactory();
 * 
 * {@link Session} session1 = xFactory.getSession("<b>mysqlx:</b>//[user1[:pwd1]@]host1[:port1]/db");
 * {@link Session} session2 = xFactory.getSession("<b>mysqlx:</b>//host2[:port2]/db?user=user2&amp;password=pwd2");
 * </pre>
 *
 */
public class SessionFactory {
    /**
     * Parses the connection string URL.
     * 
     * @param url
     *            the connection string URL.
     * @return a {@link ConnectionUrl} instance containing the URL components.
     */
    private ConnectionUrl parseUrl(String url) {
        ConnectionUrl connUrl = ConnectionUrl.getConnectionUrlInstance(url, null);
        if (connUrl.getType() != ConnectionUrl.Type.XDEVAPI_SESSION) {
            throw ExceptionFactory.createException(InvalidConnectionAttributeException.class, "Initialization via URL failed for \"" + url + "\"");
        }
        return connUrl;
    }

    /**
     * Creates {@link Session} by given URL.
     * 
     * @param url
     *            the session URL.
     * @return a {@link Session} instance.
     */
    public Session getSession(String url) {
        CJCommunicationsException latestException = null;
        ConnectionUrl connUrl = parseUrl(url);
        for (HostInfo hi : connUrl.getHostsList()) {
            try {
                return new SessionImpl(hi.exposeAsProperties());
            } catch (CJCommunicationsException e) {
                latestException = e;
            }
        }
        if (latestException != null) {
            throw latestException;
        }
        return null;
    }

    /**
     * Creates a {@link Session} using the information contained in the given properties.
     * 
     * @param properties
     *            the {@link Properties} instance that contains the session components.
     * @return a {@link Session} instance.
     */
    public Session getSession(Properties properties) {
        return new SessionImpl(properties);
    }
}
