/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Copyright 1997 - July 2008 CWI, August 2008 - 2018 MonetDB B.V.
 */

package nl.cwi.monetdb.embedded.env;

import java.io.*;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Locale;

/**
 * Class responsible to load MonetDB's embedded library. On the resources directory of the JAR inside the libs directory
 * there is present a library file for each of the operating systems MonetDBLite-Java is compiled for.
 *
 * Currently we support Windows NT, Mac OS X and Linux for x86_64.
 *
 * @author <a href="mailto:pedro.ferreira@monetdbsolutions.com">Pedro Ferreira</a>
 */
public final class MonetDBJavaLiteLoader {

	private MonetDBJavaLiteLoader() {}

	/**
	 * Operating Systems supported by MonetDBLite-Java.
	 */
	private enum SupportedOS {
		WindowsNT,
		MacOSX,
		Linux
	}

	/**
	 * Architectures supported by MonetDBLite-Java.
	 */
	private enum SupportedArc {
		X86_X64,
		ARM64 //Not yet used
	}

	/**
	 * Retrieve native library file name from the underlying operating system naming convention.
	 */
	private static String getLibraryFileName(SupportedOS os) throws MonetDBEmbeddedException {
		switch(os) {
			case WindowsNT:
				return "libmonetdblitejava.dll";
			case MacOSX:
			case Linux:
				return "libmonetdblitejava.so";
		}
		throw new MonetDBEmbeddedException("Unknown operating system");
	}

	/**
	 * Retrieve native library file base path depending on the underlying operating system and architecture.
	 */
	private static String getLibraryFileBasePath(SupportedOS os, SupportedArc arc) throws MonetDBEmbeddedException {
		switch(os) {
			case WindowsNT:
				if(arc != SupportedArc.X86_X64) {
					throw new MonetDBEmbeddedException("Architecture not supported on Windows NT");
				}
				return "/libs/windows/x86x64/";
			case MacOSX:
				if(arc != SupportedArc.X86_X64) {
					throw new MonetDBEmbeddedException("Architecture not supported on MacOS X");
				}
				return "/libs/macosx/x86x64/";
			case Linux:
				if(arc != SupportedArc.X86_X64/* && arc != SupportedArc.ARM64*/) {
					throw new MonetDBEmbeddedException("Architecture not supported on Linux");
				}
				return "/libs/linux/" + (arc == SupportedArc.X86_X64 ? "x86x64" : "arm64") + "/";
		}
		throw new MonetDBEmbeddedException("Unknown operating system");
	}

	/**
	 * The current loaded library's path.
	 */
	private static String loadedLibraryFullPath = null;

	/**
	 * The temporary directory to load the embedded library.
	 */
	private static final String tempDirectory = new File(System.getProperty("java.io.tmpdir")).getAbsolutePath();

	/**
	 * The String representation of the latest version of the MonetDB JDBC driver.
	 */
	private static final String monetDBJDBCDriverString = "2-39";

	static {
		try {
			Class.forName("nl.cwi.monetdb.jdbc.MonetDriver");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			System.err.println("monetdb-jdbc-new driver not found, add monetdb-jdbc-new jar to the CLASSPATH");
			System.exit(1);
		}
	}

	/**
	 * Check if MonetDBLite-Java native library was loaded or not.
	 *
	 * @return A boolean indicating if the MonetDBLite-Java native library was loaded or not
	 */
	public static synchronized boolean isNativeLibraryLoaded() {
		return loadedLibraryFullPath != null;
	}

	/**
	 * Detects the operating system from the os.name System property.
	 *
	 * @return The operating system found
	 * @throws MonetDBEmbeddedException If the running operative system could not be detected
	 */
	private static SupportedOS detectRunningOperatingSystem() throws MonetDBEmbeddedException {
		String OS = System.getProperty("os.name", "generic").toLowerCase(Locale.ENGLISH);

		//Check running OS
		//http://stackoverflow.com/questions/228477/how-do-i-programmatically-determine-operating-system-in-java#answer-18417382
		if ((OS.contains("mac")) || (OS.contains("darwin"))) {
			return SupportedOS.MacOSX;
		} else if (OS.contains("win")) {
			return SupportedOS.WindowsNT;
		} else if (OS.contains("nux")) {
			return SupportedOS.Linux;
		}
		throw new MonetDBEmbeddedException("The operating system " + OS + " is not currently supported by MonetDBLite-Java");
	}

	/**
	 * Detects the underlying CPU architecture and checks if MonetDBLite-Java supports it
	 *
	 * @return The CPU architecture found
	 * @throws MonetDBEmbeddedException If the architecture is not supported
	 */
	private static SupportedArc detectCPUArchitecture(SupportedOS os) throws MonetDBEmbeddedException {
		String arch = System.getProperty("os.arch").toLowerCase(Locale.ENGLISH);
		boolean is_x86_64, is_arm64;

		//Check CPU architecture
		if (os == SupportedOS.WindowsNT) {
			is_x86_64 = (System.getenv("ProgramFiles(x86)") != null);
			is_arm64 = false;
		} else {
			is_x86_64 = (arch.contains("amd64") || arch.contains("x64") || arch.contains("x86_64") || arch.contains("x8664"));
			is_arm64 = (arch.contains("arm64") || arch.contains("aarch64"));
		}

		if((os == SupportedOS.WindowsNT || os == SupportedOS.MacOSX) && !is_x86_64) {
			String desc = (os == SupportedOS.WindowsNT) ? "Windows" : "MacOS X";
			throw new MonetDBEmbeddedException("MonetDBLite-Java is only supported on x86_64 architectures on " + desc);
		} else if(os == SupportedOS.Linux && !is_x86_64 && !is_arm64) {
			throw new MonetDBEmbeddedException("MonetDBLite-Java is only supported on x86_64 and arm64 architectures on Linux");
		} else if(is_x86_64 && is_arm64) {
			throw new MonetDBEmbeddedException("Ambiguous architecture detected");
		}

		if(is_x86_64) {
			return SupportedArc.X86_X64;
		} else if(is_arm64) {
			return SupportedArc.ARM64;
		}
		throw new MonetDBEmbeddedException("The architecture " + arch + " is not currently supported by MonetDBLite-Java");
	}

	/**
	 * Performs a SHA-256 digest on a file and returns the digest.
	 *
	 * @param input The file's input stream
	 * @return The file's SHA-256 digest
	 * @throws MonetDBEmbeddedException If the SHA-256 algorithm fails to load
	 */
	private static byte[] makeSHA256Digest(InputStream input) throws MonetDBEmbeddedException {
		try {
			byte[] buffer = new byte[8192];
			MessageDigest md = MessageDigest.getInstance("SHA-256");
			try (DigestInputStream dis = new DigestInputStream(input, md)) {
				while (dis.read(buffer) != -1);
			}
			return md.digest();
		} catch (IOException | NoSuchAlgorithmException e) {
			throw new MonetDBEmbeddedException(e);
		}
	}

	/**
	 * Loads a native library into the JVM.
	 *
	 * @param libraryFilePath The library's path
	 * @param libraryFileName The library's name
	 * @return The loaded library full path or null if failed
	 */
	private static String loadNativeLibrary(String libraryFilePath, String libraryFileName) {
		File libPath = new File(libraryFilePath, libraryFileName);
		if (libPath.exists()) { //load the file
			System.load(new File(libraryFilePath, libraryFileName).getAbsolutePath());
			return libPath.getAbsolutePath();
		} else {
			return null;
		}
	}

	/**
	 * Checks if a native library exists in a directory, copying it there if it doesn't exist, and then loads it.
	 *
	 * @param libraryFilePath The library's path
	 * @param prefix An optional prefix to the extracted library file
	 * @param libraryFileName The library's name
	 * @param destinationDirectory The destination directory
	 * @return The loaded library full path or null if failed
	 * @throws MonetDBEmbeddedException If an IO error occurred
	 */
	private static String loadLibraryIntoDirectory(String libraryFilePath, String prefix, String libraryFileName,
												   String destinationDirectory) throws MonetDBEmbeddedException {
		try {
			String nativeLibraryFilePath = libraryFilePath + libraryFileName;
			String extractedLibFileName = prefix + libraryFileName;
			File extractedLibFile = new File(destinationDirectory, extractedLibFileName);

			if (extractedLibFile.exists()) { //if the file exists check if it needs to be updated
				byte[] digest1 = makeSHA256Digest(MonetDBJavaLiteLoader.class.getResourceAsStream(nativeLibraryFilePath));
				byte[] digest2 = makeSHA256Digest(new FileInputStream(extractedLibFile));

				if (Arrays.equals(digest1, digest2)) {
					return loadNativeLibrary(destinationDirectory, extractedLibFileName);
				} else {
					if (!extractedLibFile.delete()) { // delete the old file
						throw new IOException("failed to remove existing native library file: "
								+ extractedLibFile.getAbsolutePath());
					}
				}
			}

			// extract the file into the destination folder
			InputStream reader = MonetDBJavaLiteLoader.class.getResourceAsStream(nativeLibraryFilePath);
			FileOutputStream writer = new FileOutputStream(extractedLibFile);
			byte[] buffer = new byte[8192];
			int bytesRead;
			while ((bytesRead = reader.read(buffer)) != -1) {
				writer.write(buffer, 0, bytesRead);
			}
			writer.close();
			reader.close();
			return loadNativeLibrary(destinationDirectory, extractedLibFileName);
		} catch (IOException ex) {
			throw new MonetDBEmbeddedException(ex);
		}
	}

	/**
	 * Loads MonetDBLite-Java native library into the JVM if it hasn't been loaded yet.
	 *
	 * @throws MonetDBEmbeddedException If an IO error occurred
	 */
	static synchronized void loadMonetDBJavaLite() throws MonetDBEmbeddedException {
		if (loadedLibraryFullPath != null) {
			return;
		}

		SupportedOS foundOS = detectRunningOperatingSystem();
		SupportedArc foundArc = detectCPUArchitecture(foundOS);
		String nativeLibraryPath = getLibraryFileBasePath(foundOS, foundArc);
		String nativeLibraryName = getLibraryFileName(foundOS);

		if (MonetDBJavaLiteLoader.class.getResource(nativeLibraryPath + nativeLibraryName) == null) {
			loadedLibraryFullPath = null;
			throw new MonetDBEmbeddedException("Error while loading library: " + nativeLibraryPath + nativeLibraryName);
		}

		// Extract the libraries from the jar
		if(foundOS == SupportedOS.WindowsNT) {
			//On Windows we have to extract the Visual C/C++ Runtime library dependency
			//Check https://msdn.microsoft.com/en-us/library/ms235299.aspx for details
			loadLibraryIntoDirectory(nativeLibraryPath, "", "vcruntime140.dll", tempDirectory);
		}
		String prefix = "MonetDBJavaLite-" + monetDBJDBCDriverString + "-";
		loadedLibraryFullPath = loadLibraryIntoDirectory(nativeLibraryPath, prefix, nativeLibraryName, tempDirectory);
	}
}
