/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Copyright 1997 - July 2008 CWI, August 2008 - 2018 MonetDB B.V.
 */

package nl.cwi.monetdb.embedded.env;

import java.sql.Savepoint;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * The representation of a savepoint, which is a point within the current
 * transaction that can be referenced from the Connection.rollback method.
 * When a transaction is rolled back to a savepoint all changes made after
 * that savepoint are undone.
 * Savepoints can be either named or unnamed. Unnamed savepoints are
 * identified by an ID generated by the underlying data source.
 *
 * This little class is nothing more than a container for a name and/or
 * an id. Each instance of this class always has an id, which is used for
 * internal representation of the save point.
 *
 * Because the IDs which get generated are a logical sequence, application
 * wide, two concurrent transactions are guaranteed to not to have the same
 * save point identifiers. In this implementation the validaty of save points
 * is determined by the server, which makes this a light implementation.
 *
 * @author <a href="mailto:pedro.ferreira@monetdbsolutions.com">Pedro Ferreira</a>, Fabian Groffen
 * @version 1.0
 */
public class MonetDBEmbeddedSavePoint implements Savepoint {

	/**  A AtomicInteger instance to generate savepoints ids. */
	private static final AtomicInteger SavepointCounter = new AtomicInteger();

	/**
	 * Returns the next id, which is larger than the last returned id. This
	 * method is synchronized to prevent race conditions. Since this is static
	 * code, this method is shared by requests from multiple Connection objects.
	 * Therefore two successive calls to this method need not to have a
	 * difference of 1.
	 *
	 * @return the next int which is guaranteed to be higher than the one at the last call to this method.
	 */
	private static int getNextId() {
		return SavepointCounter.incrementAndGet();
	}

	/** The id of this Savepoint */
	private final int id;
	/** The name of this Savepoint */
	private final String name;

	public MonetDBEmbeddedSavePoint(String name) throws IllegalArgumentException {
		this.id = MonetDBEmbeddedSavePoint.getNextId();
		if (name == null)
			this.name = "MonetDBSP" + id;
		else
			this.name = name;
	}

	public MonetDBEmbeddedSavePoint() {
		this.id = MonetDBEmbeddedSavePoint.getNextId();
		this.name = "MonetDBSP" + id;
	}

	/**
	 * Retrieves the generated ID for the savepoint that this Savepoint object represents.
	 *
	 * @return the numeric ID of this savepoint
	 * @throws MonetDBEmbeddedException if this is a named savepoint
	 */
	@Override
	public int getSavepointId() throws MonetDBEmbeddedException {
		return id;
	}

	/**
	 * Retrieves the name of the savepoint that this Savepoint object represents.
	 *
	 * @return the name of this savepoint
	 * @throws MonetDBEmbeddedException if this is an un-named savepoint
	 */
	@Override
	public String getSavepointName() throws MonetDBEmbeddedException {
		return name;
	}
}
