package global.hh.openapi.sdk.utils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import global.hh.openapi.sdk.exception.BaseException;
import global.hh.openapi.sdk.exception.JsonParseException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class JsonUtils {

    private static final ObjectMapper objectMapper = new ObjectMapper();

    static {
        //去掉默认的时间戳格式
        //objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);

        //设置为中国上海时区
        //objectMapper.setTimeZone(TimeZone.getTimeZone("Asia/Shanghai"));

        objectMapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);

        //空值不序列化
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        //反序列化时，属性不存在的兼容处理
        objectMapper.getDeserializationConfig().withoutFeatures(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        //单引号处理
        objectMapper.configure(com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
    }

    private JsonUtils() {
    }

    public static final ObjectMapper getInstance() {
        return objectMapper;
    }

    public static String toJson(Object obj) throws BaseException {

        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new JsonParseException();
        }
    }

    public static <T> T toObject(String json, Class<T> clazz) throws BaseException {
        try {
            return objectMapper.readValue(json, clazz);
        } catch (IOException e) {
            throw new JsonParseException();
        }
    }

    public static <T> T toObject(String json, JavaType javaType) throws BaseException {
        try {
            return objectMapper.readValue(json, javaType);
        } catch (IOException e) {
            throw new JsonParseException();
        }
    }

    public static Map<String, Object> toMap(String json) throws BaseException {
        try {
            return objectMapper.readValue(json, Map.class);
        } catch (IOException e) {
            throw new JsonParseException();
        }
    }

    public static <T> Map<String, T> toMap(String json, Class<T> clazz) throws BaseException {
        Map<String, Map<String, Object>> map;
        try {
            map = objectMapper.readValue(json,
                    new TypeReference<Map<String, T>>() {
                    });
        } catch (IOException e) {
            throw new JsonParseException();
        }
        Map<String, T> result = new HashMap<String, T>();
        for (Map.Entry<String, Map<String, Object>> entry : map.entrySet()) {
            result.put(entry.getKey(), mapToObject(entry.getValue(), clazz));
        }
        return result;
    }

    public static <T> List<T> toList(String json, Class<T> clazz) throws BaseException {
        List<Map<String, Object>> list = null;
        try {
            list = objectMapper.readValue(json,
                    new TypeReference<List<T>>() {
                    });
        } catch (IOException e) {
            throw new JsonParseException();
        }
        List<T> result = new ArrayList<T>();
        for (Map<String, Object> map : list) {
            result.add(mapToObject(map, clazz));
        }
        return result;
    }

    public static <T> T mapToObject(Map map, Class<T> clazz) {
        return objectMapper.convertValue(map, clazz);
    }
}