package eu.vaadinonkotlin.vaadin8

import com.github.mvysny.karibudsl.v8.fillParent
import com.github.mvysny.karibudsl.v8.getColumnBy
import com.github.mvysny.karibudsl.v8.perc
import com.github.mvysny.karibudsl.v8.w
import eu.vaadinonkotlin.FilterFactory
import eu.vaadinonkotlin.Listeners
import eu.vaadinonkotlin.listeners
import com.vaadin.data.BeanPropertySet
import com.vaadin.data.HasValue
import com.vaadin.data.PropertyDefinition
import com.vaadin.data.provider.ConfigurableFilterDataProvider
import com.vaadin.server.SerializableConsumer
import com.vaadin.shared.Registration
import com.vaadin.shared.ui.ValueChangeMode
import com.vaadin.ui.Component
import com.vaadin.ui.Grid
import com.vaadin.ui.HasValueChangeMode
import com.vaadin.ui.components.grid.HeaderRow
import java.io.Serializable
import kotlin.reflect.KClass
import kotlin.reflect.KProperty1
import kotlin.streams.toList

/**
 * Maintains a set of fields - monitors bound fields for values and provides [filter] of type F.
 * Fires [onFilterChangeListeners] on every filter change.
 * @param T the type of items in the grid.
 * @param F the type of the filter
 * @param filterFieldFactory used to create filters from filter components by invoking [FilterFieldFactory.createFilter]
 */
public class FilterBinder<T : Any, F : Any>(
        public val filterFieldFactory: FilterFieldFactory<T, F>,
        public val filterFactory: FilterFactory<F>
) : Serializable {

    /**
     * The current filter as generated by [filterFieldFactory] from the most current states of all bound filter fields.
     * Updated on every filter field change.
     */
    public var filter: F? = null
        private set

    /**
     * Invoked when the [filter] changes.
     */
    public val onFilterChangeListeners: Listeners<SerializableConsumer<F?>> = listeners()

    private val filterComponents = mutableMapOf<HasValue<*>, FilterFieldWatcher<*>>()

    /**
     * Binds given filtering field to a container - starts filtering based on the contents of the field, and starts watching for field value changes.
     * Does not add the component to the [headerRow].
     * @param field The field which provides the filtering values, not null. [FilterFieldFactory.createFilter] is used to convert
     * the field's value to a filter.
     * @param property The bean property on which the filtering will be performed, not null.
     */
    public fun <V> bind(field: HasValue<V?>, property: PropertyDefinition<T, V?>) {
        unbind(field)
        val filterFieldWatcher = FilterFieldWatcher(field, property)
        filterFieldWatcher.registration = field.addValueChangeListener(filterFieldWatcher)
        filterComponents[field] = filterFieldWatcher
    }

    public fun unbind(field: HasValue<*>) {
        filterComponents.remove(field)?.unregisterWatcher()
    }

    public fun unbindAll() {
        filterComponents.keys.toList().forEach { unbind(it) }
    }

    /**
     * Listens on value change on given field and updates [ConfigurableFilterDataProvider.setFilter] accordingly.
     * @property field The field which provides the filtering values.
     * @property property The bean property on which the filtering will be performed.
     * @param V the value type
     */
    private inner class FilterFieldWatcher<V>(private val field: HasValue<V?>, private val property: PropertyDefinition<T, V?>) :
            HasValue.ValueChangeListener<V?> {

        /**
         * The current container filter, may be null if no filtering is currently needed because the
         * field's value indicates that the filtering is disabled for this column (e.g. the text filter is blank, the filter field is cleared, etc).
         */
        var currentFilter: F? = null

        init {
            valueChange()
        }

        override fun valueChange(event: HasValue.ValueChangeEvent<V?>) {
            valueChange()
        }

        private fun valueChange(value: V? = field.value) {
            val newFilter = filterFieldFactory.createFilter(value, field, property)
            setNewFilter(newFilter)
        }

        private fun setNewFilter(newFilter: F?) {
            if (newFilter != currentFilter) {
                currentFilter = newFilter
                recomputeFilter()
            }
        }

        fun unregisterWatcher() {
            setNewFilter(null)
            registration.remove()
        }

        lateinit var registration: Registration
    }

    private fun recomputeFilter() {
        val filters: Set<F> = filterComponents.values.mapNotNull { it.currentFilter }.toSet()
        filter = filterFactory.and(filters)
        onFilterChangeListeners.fire.accept(filter)
    }
}

/**
 * Wraps [HeaderRow] and uses [FilterBinder] to track filter UI components for changes. After the user
 * changes the value in the filter UI component, a new Grid filter of type [F] is computed and
 * set to [Grid.getDataProvider].
 *
 * Every filter component is configured using the [componentConfigurator] closure.
 * @param T the type of items in the grid.
 * @param F the type of the filters accepted by grid's [ConfigurableFilterDataProvider].
 * @param grid the owner grid. It is expected that [Grid.getDataProvider] is of type [VokDataProvider]<T>
 * (or [ConfigurableFilterDataProvider]<T, F, F>).
 * @property itemClass the type of items shown by the Grid.
 * @property headerRow the wrapped header row
 * @property filterFieldFactory used to automatically create filter UI components
 * for bean properties, and to create Grid filters of type [F].
 * @param filterFactory used to combine filter values when multiple filters are applied
 * (using the [FilterFactory.and] function).
 * @property componentConfigurator invoked for every filter component when created. By default every component
 * is set to 100%, and by default this closure
 * will do nothing.
 */
@Suppress("UNCHECKED_CAST")
public class FilterRow<T : Any, F : Any>(
        public val grid: Grid<T>,
        public val itemClass: Class<T>,
        public val headerRow: HeaderRow,
        public val filterFieldFactory: FilterFieldFactory<T, F>,
        public val filterFactory: FilterFactory<F>,
        private val componentConfigurator: (filterComponent: HasValue<*>, property: PropertyDefinition<T, *>) -> Unit = { _, _ -> }
) : Serializable {

    /**
     * Binds filters from this row. Whenever a bound filter changes, it is set to [Grid.getDataProvider].
     */
    public val binder: FilterBinder<T, F> = FilterBinder(filterFieldFactory, filterFactory)

    init {
        binder.onFilterChangeListeners.add(SerializableConsumer { filter ->
            (grid.dataProvider as ConfigurableFilterDataProvider<T, F, F>).setFilter(filter)
        })
    }

    /**
     * Invoked when the filter changes.
     */
    public val onFilterChangeListeners: Listeners<SerializableConsumer<F?>>
        get() = binder.onFilterChangeListeners

    /**
     * Re-generates all filter components in this header row. Removes all old filter components and
     * creates a new set and populates them into the [headerRow].
     */
    public fun generateFilterComponents(valueChangeMode: ValueChangeMode = ValueChangeMode.LAZY) {
        clear()

        val properties: Map<String, PropertyDefinition<T, *>> =
                BeanPropertySet.get(itemClass).properties.toList().associateBy { it.name }
        for (propertyId in grid.columns.mapNotNull { it.id }) {
            val property: PropertyDefinition<T, *>? = properties[propertyId]
            val field: HasValue<*>? = if (property == null) null else filterFieldFactory.createField(property as PropertyDefinition<T, Any?>)
            (field as? HasValueChangeMode)?.valueChangeMode = valueChangeMode
            val cell = headerRow.getCell(propertyId)
            if (field != null) {
                binder.bind(field as HasValue<Any?>, property!! as PropertyDefinition<T, Any?>)
                cell.component = field as Component
            }
        }
    }

    /**
     * Configures every Vaadin UI filter [field]. By default the width is set to 100%.
     * Calls [componentConfigurator].
     */
    private fun configureField(field: HasValue<*>, property: PropertyDefinition<T, *>) {
        (field as Component).w = fillParent
        componentConfigurator(field, property)
    }

    /**
     * Unbinds all filters and removes all filter components from this [headerRow].
     */
    public fun clear() {
        binder.unbindAll()
        grid.columns.forEach { column ->
            // can't set cell.component to null, that would fail with NPE!
            headerRow.getCell(column).text = "" // this also removes the cell field from the header row
        }
    }

    /**
     * [clear]s and removes the [headerRow] from the grid.
     */
    public fun remove() {
        clear()
        grid.removeHeaderRow(headerRow)
    }

    /**
     * Returns the filter component filtering given [property]. Fails if no filter component has been generated for
     * that property.
     */
    public fun getFilterComponent(property: KProperty1<T, *>): HasValue<Any?> {
        val column: Grid.Column<T, Any?> = grid.getColumnBy(property)
        val component: Component = headerRow.getCell(column).component
                ?: throw IllegalArgumentException("There is no filter configured for $property")
        return component as HasValue<Any?>
    }
}

/**
 * Re-creates filters in this header row. Simply call `grid.appendHeaderRow().generateFilterComponents(grid)` to automatically attach
 * filters to non-generated columns. Please note that filters are not re-generated when the container data source is changed.
 *
 * Only use this function if you wish to use your custom filter hierarchy. However, typically you use [VokDataProvider] which
 * uses `vok-dataprovider` filters and then you should use _the other_ `generateFilterComponents()` function.
 * @param T the type of items in the grid.
 * @param grid the owner grid.
 * @param filterFieldFactory used to create the filters themselves. If null, [DefaultFilterFieldFactory] is used.
 * @param valueChangeMode how eagerly to apply the filtering after the user changes the filter value. Only applied to [HasValueChangeMode];
 * typically only applies to inline filter
 * components (most importantly [com.vaadin.ui.TextField]), typically ignored for popup components (such as [com.github.vok.framework.NumberFilterPopup])
 * where the values are applied after the user clicks the "Apply" button. Defaults to [ValueChangeMode.LAZY].
 * @param componentConfigurator invoked for every filter component when created. By default every component
 * is set to 100%, and by default this closure
 * will do nothing.
 * @return the [FilterRow] row with filter components already generated.
 */
@Suppress("UNCHECKED_CAST")
public fun <T : Any, F : Any> HeaderRow.generateFilterComponents(
        grid: Grid<T>, itemClass: KClass<T>,
        filterFactory: FilterFactory<F>,
        filterFieldFactory: FilterFieldFactory<T, F> = DefaultFilterFieldFactory(filterFactory),
        valueChangeMode: ValueChangeMode = ValueChangeMode.LAZY,
        componentConfigurator: (filterComponent: HasValue<*>, property: PropertyDefinition<T, *>) -> Unit = { _, _ -> }
): FilterRow<T, F> {
    val filterRow: FilterRow<T, F> = FilterRow(grid, itemClass.java, this,
            filterFieldFactory, filterFactory, componentConfigurator)
    filterRow.generateFilterComponents(valueChangeMode)
    return filterRow
}
