package eu.toop.playground.demoui.dpweb;

import static io.javalin.plugin.rendering.template.TemplateUtil.model;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.helger.commons.error.level.EErrorLevel;
import com.helger.commons.mime.CMimeType;
import com.helger.commons.mime.MimeTypeDeterminator;
import com.typesafe.config.ConfigFactory;

import eu.toop.connector.api.me.EMEProtocol;
import eu.toop.connector.api.me.incoming.IIncomingEDMResponse;
import eu.toop.connector.api.me.incoming.IMEIncomingTransportMetadata;
import eu.toop.connector.api.me.incoming.IncomingEDMErrorResponse;
import eu.toop.connector.api.me.incoming.IncomingEDMResponse;
import eu.toop.connector.api.me.incoming.MEIncomingTransportMetadata;
import eu.toop.connector.api.me.model.MEPayload;
import eu.toop.connector.api.rest.TCIncomingMessage;
import eu.toop.connector.api.rest.TCIncomingMetadata;
import eu.toop.connector.api.rest.TCOutgoingMessage;
import eu.toop.connector.api.rest.TCOutgoingMetadata;
import eu.toop.connector.api.rest.TCPayload;
import eu.toop.connector.api.rest.TCRestJAXB;
import eu.toop.edm.EDMErrorResponse;
import eu.toop.edm.EDMRequest;
import eu.toop.edm.EDMResponse;
import eu.toop.edm.IEDMTopLevelObject;
import eu.toop.edm.xml.EDMPayloadDeterminator;
import eu.toop.kafkaclient.ToopKafkaClient;
import eu.toop.kafkaclient.ToopKafkaSettings;
import eu.toop.playground.dp.DPException;
import eu.toop.playground.dp.model.Attachment;
import eu.toop.playground.dp.model.EDMResponseWithAttachment;
import eu.toop.playground.dp.service.ToopDP;
import io.javalin.Javalin;
import io.javalin.http.Context;

public class DpApp {
  public static final DpConfig APPCONFIG =
      new DpConfig(
          ConfigFactory.load()
              .withFallback(ConfigFactory.parseFile(Paths.get("demoui-dp.conf").toFile()))
              .withFallback(ConfigFactory.parseResources("demoui-dp.conf"))
              .resolve());
  private static final Logger LOGGER = LoggerFactory.getLogger(DpApp.class);
  private static ToopDP miniDP =
      new ToopDP(APPCONFIG.getDp().getDatasetGBM(), APPCONFIG.getDp().getDatasetDocument());

  public static void main(String[] args) throws IOException {
    if (APPCONFIG.getDp().getKafka().enabled) {
      ToopKafkaSettings.setKafkaEnabled(true);
      ToopKafkaSettings.defaultProperties()
          .put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, APPCONFIG.getDp().getKafka().url);
      ToopKafkaSettings.setKafkaTopic(APPCONFIG.getDp().getKafka().topic);
    } else ToopKafkaSettings.setKafkaEnabled(false);

    Javalin.create(
            javalinConfig -> {
              if (APPCONFIG.getServer().cors) javalinConfig.enableCorsForAllOrigins();
            })
        .start(APPCONFIG.getServer().getPort())
        .get("/", DpApp::index)
        .post(
            APPCONFIG.getDp().getDirect().getReceive(),
            context -> {
              context.contentType(CMimeType.APPLICATION_XML.getAsString());
              byte[] responseBytes;
              try {
                responseBytes = miniDP.createXMLResponseFromRequest(context.bodyAsBytes());
              } catch (DPException e) {
                responseBytes = e.getEdmErrorResponse().getWriter().getAsBytes();
              }
              LOGGER.info(
                  "Sent response with status {}",
                  sendResponse(responseBytes, APPCONFIG.getDp().getDirect().getSubmit()));
            })
        .post(
            APPCONFIG.getDp().getToop().getReceive(),
            context -> {
              final String incomingMessage = context.body();
              final TCIncomingMessage tcIncomingMessage =
                  TCRestJAXB.incomingMessage().read(incomingMessage);
              int statusCode = createAndSendTCOutgoingMessageFromIncoming(tcIncomingMessage);
              if (statusCode == 0) LOGGER.info("Sent no response.");
              else LOGGER.info("Sent response with status {}", statusCode);
            })
        .get("/datasets", context -> context.result(displayDatasets()))
        .get(
            "/reload",
            context -> {
              miniDP =
                  new ToopDP(
                      APPCONFIG.getDp().getDatasetGBM(), APPCONFIG.getDp().getDatasetDocument());
              context.result(displayDatasets());
            });
  }

  private static void index(Context ctx) {
    ctx.render("/templates/index.vm", model("config", APPCONFIG));
  }

  private static String displayDatasets() {
    StringBuilder stringBuilder = new StringBuilder();
    miniDP
        .getRegisteredOrganizationDatasource()
        .getDatasets()
        .values()
        .forEach(s -> stringBuilder.append(s).append("\n"));
    miniDP
        .getDocumentDatasource()
        .getDatasets()
        .values()
        .forEach(s -> stringBuilder.append(s).append("\n"));
    return stringBuilder.toString();
  }

  private static int createAndSendTCOutgoingMessageFromIncoming(TCIncomingMessage tcIncomingMessage)
      throws IOException {
    final TCIncomingMetadata metadata = tcIncomingMessage.getMetadata();
    //        LOGGER.info("DP Received Metadata: " + metadata);
    if (metadata != null) {
      ToopKafkaClient.send(
          EErrorLevel.INFO,
          "Elonia DP Received Incoming message with sender " + metadata.getSenderID());
      ToopKafkaClient.send(
          EErrorLevel.INFO,
          "Elonia DP Received Incoming message with receiver " + metadata.getReceiverID());
      ToopKafkaClient.send(
          EErrorLevel.INFO,
          "Elonia DP Received Incoming message with docTypeID " + metadata.getDocTypeID());
      ToopKafkaClient.send(
          EErrorLevel.INFO,
          "Elonia DP Received Incoming message with payloadType " + metadata.getPayloadType());
      ToopKafkaClient.send(
          EErrorLevel.INFO,
          "Elonia DP Received Incoming message with processID " + metadata.getProcessID());
    } else {
      ToopKafkaClient.send(
          EErrorLevel.ERROR, "Elonia DP Received Incoming message with no metadata");
      throw new IllegalStateException("Elonia DP Received Incoming message with no metadata");
    }
    TCPayload tcPayload = tcIncomingMessage.getPayloadAtIndex(0);
    LOGGER.info("DP Received Payload Content ID: " + tcPayload.getContentID());
    LOGGER.info("DP Received Payload Mime Type: " + tcPayload.getMimeType());

    if (tcPayload.getValue() != null) {
      IEDMTopLevelObject aTLO =
          EDMPayloadDeterminator.parseAndFind(new ByteArrayInputStream(tcPayload.getValue()));

      if (aTLO instanceof EDMResponse) {
        final EDMResponse edmResponse = (EDMResponse) aTLO;
        //                LOGGER.warn("DP Received Unexpected EDMResponse");
        ToopKafkaClient.send(EErrorLevel.WARN, "Elonia DP Received an unexpected EDMResponse");
        LOGGER.info("DP Received Payload:\n {}", edmResponse.getWriter().getAsString());
        // TODO define what 0 means as an http status code ;-) At least a constant for that "0" would make sense
        return 0;
      }
      if (aTLO instanceof EDMErrorResponse) {
        final EDMErrorResponse edmErrorResponse = (EDMErrorResponse) aTLO;
        //                LOGGER.warn("DP Received EDMErrorResponse");
        ToopKafkaClient.send(
            EErrorLevel.ERROR, "Elonia DP Received an unexpected EDMErrorResponse");
        //                LOGGER.info("DP Received Payload:\n {}",
        // edmErrorResponse.getWriter().getAsString());
        ToopKafkaClient.send(
            EErrorLevel.ERROR,
            "DP Received Payload:\n" + edmErrorResponse.getWriter().getAsString());

        return 0;
      }
      if (aTLO instanceof EDMRequest) {
        final EDMRequest edmRequest = (EDMRequest) aTLO;

        String requestID = "[" + edmRequest.getRequestID() + "] ";
        ToopKafkaClient.send(EErrorLevel.INFO, requestID + "Elonia DP Received an EDMRequest");

        LOGGER.info("DP Received Payload:\n" + edmRequest.getWriter().getAsString());

        MEIncomingTransportMetadata meIncomingTransportMetadata =
            MEIncomingTransportMetadata.create(metadata);
        TCOutgoingMessage tcOutgoingMessage;

        final MEIncomingTransportMetadata aMetadataInverse =
            new MEIncomingTransportMetadata(
                meIncomingTransportMetadata.getReceiverID(),
                    meIncomingTransportMetadata.getSenderID(),
                meIncomingTransportMetadata.getDocumentTypeID(),
                    meIncomingTransportMetadata.getProcessID());
        try {
          EDMResponseWithAttachment edmResponse =
              miniDP.createEDMResponseWithAttachmentsFromRequest(edmRequest);

          ToopKafkaClient.send(
              EErrorLevel.INFO, requestID + "DP created an EDMResponse successfully");

          List<MEPayload> attachments;
          if (edmResponse.getAttachment().isPresent()) {
            Attachment attachment = edmResponse.getAttachment().get();
            byte[] fileBytes = Files.readAllBytes(attachment.getAttachedFile().toPath());
            attachments =
                Collections.singletonList(
                    MEPayload.builder()
                        .data(fileBytes)
                        .mimeType(MimeTypeDeterminator.getInstance().getMimeTypeFromBytes(fileBytes))
                            .contentID(attachment.getAttachedFileCid())
                        .build());
          } else attachments = new ArrayList<>();

          IncomingEDMResponse incomingEDMResponse =
              new IncomingEDMResponse(edmResponse.getEdmResponse(), tcPayload.getContentID (), attachments, aMetadataInverse);

          tcOutgoingMessage =
              createTCOutgoingMessageFromIncomingResponse(
                  incomingEDMResponse.getMetadata(), incomingEDMResponse);
          ToopKafkaClient.send(
              EErrorLevel.INFO, requestID + "DP created an OutgoingMessage successfully");

          int statusCode =
              sendResponse(
                  TCRestJAXB.outgoingMessage().getAsBytes(tcOutgoingMessage),
                  APPCONFIG.getDp().getToop().getSubmit() + "/response");

          EErrorLevel errorLevel;

          switch (statusCode){
            case 200:
              errorLevel = EErrorLevel.INFO;
              break;
            case 400:
              errorLevel = EErrorLevel.ERROR;
              break;
            default:
              errorLevel = EErrorLevel.WARN;
          }

          ToopKafkaClient.send(
                  errorLevel,
                  requestID + "Elonia DP pushed response to connector with status code " + statusCode);

          return statusCode;

        } catch (DPException e) {
          EDMErrorResponse edmErrorResponse = e.getEdmErrorResponse();

          ToopKafkaClient.send(
              EErrorLevel.ERROR,
              requestID
                  + "Elonia DP failed to create an EDMResponse,"
                  + " sending back an EDMErrorResponse to DC");
          ToopKafkaClient.send(
              EErrorLevel.ERROR, requestID + "Error was: \"" + e.getMessage() + "\"");
          //                    ToopKafkaClient.send(EErrorLevel.ERROR,requestID +
          // "EDMErrorResponse:\n "+edmErrorResponse.getWriter().getAsString());

          IncomingEDMErrorResponse incomingEDMResponse =
              new IncomingEDMErrorResponse(edmErrorResponse, tcPayload.getContentID (), aMetadataInverse);

          tcOutgoingMessage =
              createTCOutgoingMessageFromIncomingResponse(
                  incomingEDMResponse.getMetadata(), incomingEDMResponse);

          ToopKafkaClient.send(
              EErrorLevel.INFO, requestID + "Elonia DP created an OutgoingMessage successfully");
          int statusCode =
              sendResponse(
                  TCRestJAXB.outgoingMessage().getAsBytes(tcOutgoingMessage),
                  APPCONFIG.getDp().getToop().getSubmit() + "/error");

          EErrorLevel errorLevel;

          switch (statusCode){
            // TODO use constants
            case 200:
              errorLevel = EErrorLevel.INFO;
              break;
            case 400:
              errorLevel = EErrorLevel.ERROR;
              break;
            default:
              errorLevel = EErrorLevel.WARN;
          }

          ToopKafkaClient.send(
                  errorLevel,
                  requestID + "Elonia DP pushed response to connector with status code " + statusCode);

          return statusCode;
        }
      }
    }
    return 0;
  }

  private static TCOutgoingMessage createTCOutgoingMessageFromIncomingResponse(
      IMEIncomingTransportMetadata metadata, IIncomingEDMResponse incomingResponse) {
    final TCOutgoingMessage aOM = new TCOutgoingMessage();
    {
      final TCOutgoingMetadata aMetadata = new TCOutgoingMetadata();
      // invert sender and receiver
      aMetadata.setReceiverID(
          TCRestJAXB.createTCID(
              metadata.getReceiverID().getScheme(), metadata.getReceiverID().getValue()));
      aMetadata.setSenderID(
          TCRestJAXB.createTCID(
              metadata.getSenderID().getScheme(), metadata.getSenderID().getValue()));
      aMetadata.setDocTypeID(
          TCRestJAXB.createTCID("toop-doctypeid-qns", "QueryResponse::toop-edm:v2.0"));
      aMetadata.setProcessID(
          TCRestJAXB.createTCID(
              metadata.getProcessID().getScheme(), metadata.getProcessID().getValue()));
      aMetadata.setTransportProtocol(EMEProtocol.AS4.getTransportProfileID());
      aOM.setMetadata(aMetadata);
    }

    {
      final TCPayload aPayload = new TCPayload();
      TCPayload filePayload = null;
      byte[] payload = null;

      if (incomingResponse instanceof IncomingEDMResponse) {
          payload = ((IncomingEDMResponse) incomingResponse).getResponse().getWriter().getAsBytes();

          if(((IncomingEDMResponse) incomingResponse).attachments().size()>0) {
            filePayload = new TCPayload();
            MEPayload attachedFile = ((IncomingEDMResponse) incomingResponse).attachments().getLastValue();
            filePayload.setContentID(attachedFile.getContentID()+"@elonia-dev");
            filePayload.setMimeType(attachedFile.getMimeTypeString());
            filePayload.setValue(attachedFile.getData().bytes());
          }
        aPayload.setContentID(((IncomingEDMResponse) incomingResponse).getResponse().getRequestID()+"@elonia-dev");
      }
      if (incomingResponse instanceof IncomingEDMErrorResponse) {
        payload =
                ((IncomingEDMErrorResponse) incomingResponse)
                        .getErrorResponse()
                        .getWriter()
                        .getAsBytes();
        aPayload.setContentID(((IncomingEDMErrorResponse) incomingResponse).getErrorResponse().getRequestID()+"@elonia-dev");

      }

      aPayload.setValue(payload);
      aPayload.setMimeType(CMimeType.APPLICATION_XML.getAsString());
      aOM.addPayload(aPayload);
      if(filePayload != null){
        aOM.addPayload(filePayload);
      }
    }
    return aOM;
  }

  private static int sendResponse(byte[] responseMessage, String url) throws IOException {
    Objects.requireNonNull(responseMessage, "The supplied Response must not be null!");
    try (CloseableHttpClient client = HttpClients.createDefault()) {
      HttpPost httpPost = new HttpPost(url);
      httpPost.setHeader(HttpHeaders.CONTENT_TYPE, CMimeType.APPLICATION_XML.getAsString());
      ByteArrayEntity myEntity = new ByteArrayEntity(responseMessage);

      httpPost.setEntity(myEntity);
      CloseableHttpResponse response = client.execute(httpPost);
      String responseBody =
          new BufferedReader(new InputStreamReader(response.getEntity().getContent()))
              .lines()
              .parallel()
              .collect(Collectors.joining("\n"));
      int statusCode = response.getStatusLine().getStatusCode();
      LOGGER.info("DP got response when sending message:\n {}", responseBody);
      if(statusCode == 400)
        ToopKafkaClient.send(EErrorLevel.ERROR,"Response from connector:\n "+responseBody);
      return statusCode;
    }
  }
}
