package eu.tarienna.jmshelper;

import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.core.MessageCreator;
import org.springframework.util.Assert;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;
import javax.jms.TextMessage;
import java.util.UUID;

/**
 * Helper to send message and receive replies.
 */
public class JmsMessageSenderImpl implements JmsMessageSender {

    private final JmsTemplate jmsTemplate;

    /**
     * Construct a new message sender object.
     *
     * @param jmsTemplate the {@link JmsTemplate} - must not be null
     */
    public JmsMessageSenderImpl(final JmsTemplate jmsTemplate) {
        Assert.notNull(jmsTemplate);
        this.jmsTemplate = jmsTemplate;
    }

    @Override
    public String sendMessageAndReceiveReply(final String text, final String requestQueue,
                                             final String responseQueue) throws JMSException {
        Assert.notNull(text);
        Assert.notNull(requestQueue);
        Assert.notNull(responseQueue);
        final String correlationId = createCorrelationId();
        final MessageCreator messageCreator = session -> createMessage(session, text, responseQueue, correlationId);
        jmsTemplate.send(requestQueue, messageCreator);
        return receiveResponse(responseQueue, correlationId);
    }

    @Override
    public void sendMessageAndReceiveReply(final String text, final String requestQueue,
                                           final String responseQueue, final AsyncReceiver asyncReceiver) {
        Assert.notNull(text);
        Assert.notNull(requestQueue);
        Assert.notNull(responseQueue);
        Assert.notNull(asyncReceiver);
        final String correlationId = createCorrelationId();
        final MessageCreator messageCreator = session -> createMessage(session, text, responseQueue, correlationId);
        jmsTemplate.send(requestQueue, messageCreator);
        new Thread(() -> {
            try {
                asyncReceiver.receiveReply(receiveResponse(responseQueue, correlationId));
            } catch (JMSException e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            }
        }).start();
    }

    @Override
    public void sendMessage(final String text, final String requestQueue) {
        Assert.notNull(text);
        Assert.notNull(requestQueue);
        final MessageCreator messageCreator = session -> createMessage(session, text);
        jmsTemplate.send(requestQueue, messageCreator);
    }

    private String createCorrelationId() {
        return UUID.randomUUID().toString();
    }

    private Message createMessage(final Session session, final String text,
                                  final String responseQueue, final String correlationId) throws JMSException {
        final Message message = session.createTextMessage(text);
        message.setJMSCorrelationID(correlationId);
        message.setJMSReplyTo(session.createQueue(responseQueue));
        return message;
    }

    private Message createMessage(final Session session, final String text) throws JMSException {
        return session.createTextMessage(text);
    }

    private String createSelectorFrom(final String correlationId) {
        return "JMSCorrelationID='" + correlationId + "'";
    }

    private String receiveResponse(final String responseQueue, final String correlationId) throws JMSException {
        final String selector = createSelectorFrom(correlationId);
        final Message message = jmsTemplate.receiveSelected(responseQueue, selector);
        if (message != null && message instanceof TextMessage) {
            final TextMessage textMessage = (TextMessage) message;
            return textMessage.getText();
        }
        return null;
    }

}
