package eu.shiftforward.adstax.util

import java.util.UUID
import java.util.concurrent.TimeoutException

import scala.concurrent.duration._
import scala.concurrent.{ ExecutionContext, Future, Promise }

import akka.actor._
import akka.pattern.ask
import com.github.sstone.amqp.Amqp._
import com.github.sstone.amqp.RpcServer.{ IProcessor, ProcessResult }
import com.github.sstone.amqp._
import com.rabbitmq.client.AMQP.Queue
import com.rabbitmq.client.ConnectionFactory

import eu.shiftforward.adstax.config

/**
 * Inheriting from this trait provides useful methods to interact with RabbitMQ using akka Actors.
 */
@deprecated("Classes mixing in RabbitMQUtilAsync should instead use an AmqpClient instance", "0.3.0")
trait RabbitMQUtilAsync {
  def rmqActorRefFactory: ActorRefFactory
  def rmqConfig: config.RabbitMQ

  /**
   * Generates a name for an anonymous queue.
   * This function should generate ids that make debug easier, while having an
   * easily identifiable name to allow custom policies.
   * Required since the suggested official implementation (ie. "") does not work properly.
   *
   * @return a random queue name
   */
  private def generateName() = s"gen-${this.getClass.getSimpleName}-${UUID.randomUUID()}"

  /**
   * Generates a name for an anonymous client-facing queue.
   * This function should generate clean ids that can be shown to a client, while having an
   * easily identifiable name to allow custom policies.
   * Required since the suggested official implementation (ie. "") does not work properly.
   *
   * @return a random queue name
   */
  private def generateClientFacingName() = s"out-${UUID.randomUUID()}"

  private lazy val waitForRmqTimeout = rmqConfig.timeout
  private lazy val clientFacingMessageTTL = rmqConfig.cfMessageTtl.get.toMillis
  private lazy val clientFacingQueueTTL = rmqConfig.cfQueueTtl.get.toMillis

  lazy val connectionOwner = {
    val rmqHost = rmqConfig.host
    val rmqPort = rmqConfig.port
    val rmqUser = rmqConfig.username
    val rmqPass = rmqConfig.password

    val connFactory = new ConnectionFactory()
    connFactory.setHost(rmqHost)
    connFactory.setPort(rmqPort)
    connFactory.setUsername(rmqUser)
    connFactory.setPassword(rmqPass)

    rmqActorRefFactory.actorOf(ConnectionOwner.props(connFactory))
  }

  private def createChildActor(props: Props) = try {
    ConnectionOwner.createChildActor(conn = connectionOwner, channelOwner = props, timeout = waitForRmqTimeout)
  } catch {
    case t: TimeoutException => throw new TimeoutException("Could not connect to RabbitMQ")
  }

  private def createChannel(onConnected: ActorRef => Unit): ActorRef = {
    val channel = createChildActor(ChannelOwner.props())

    Amqp.onConnection(rmqActorRefFactory, channel, () => onConnected(channel))
    channel
  }

  private def createConsumer(actor: ActorRef, onConnected: ActorRef => Unit): ActorRef = {
    val consumer = createChildActor(Consumer.props(
      listener = actor,
      autoack = true,
      channelParams = Some(ChannelParameters(1))))

    Amqp.onConnection(rmqActorRefFactory, consumer, () => onConnected(consumer))
    consumer
  }

  private def rpcActor(props: Props, onConnected: ActorRef => Unit): ActorRef = {
    val rpcActor = createChildActor(props)

    Amqp.onConnection(rmqActorRefFactory, rpcActor, () => onConnected(rpcActor))
    rpcActor
  }

  /**
   * Declare an exchange asynchronously.
   * When the broker has a connection to the channel, the exchange will be declared.
   *
   * @param exchangeName the exchange name
   * @param exchangeType the exchange type
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @return the future that completes this exchange declaration
   */
  def declareExchange(exchangeName: String, exchangeType: String = "topic",
    handlerActor: Option[ActorRef] = None): Future[Unit] = {
    val p = Promise[Unit]()

    val declareExchange =
      DeclareExchange(ExchangeParameters(exchangeName, passive = false, exchangeType, durable = true))

    createChannel(channel => {

      val proxy = rmqActorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case ok @ Ok(_: DeclareExchange, _) =>
            p.trySuccess(())
            handlerActor.foreach(channel ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p.tryFailure(error.reason)
        }
      }))

      channel.tell(Record(declareExchange), proxy.actorRef)
    })
    p.future
  }

  /**
   * Creates a producer.
   * When the broker has a connection to the channel, the handler actor will be notified that the producer is connected.
   *
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @return the actor ref of the producer
   */
  def createProducer(handlerActor: Option[ActorRef]): ActorRef = {
    val producer = createChildActor(ChannelOwner.props())

    Amqp.onConnection(rmqActorRefFactory, producer, () => {
      handlerActor.foreach(producer ! AddStatusListener(_))
    })

    producer
  }

  /**
   * Creates a producer which stashes messages until the producer is fully connected to AMQP.
   *
   * @return the actor ref of the producer proxy.
   */
  def createStashedProducer(): ActorRef = {
    rmqActorRefFactory.actorOf(Props(new BaseAmqpProducerStashActor {
      def setupAmqp = Future.successful(createProducer(Some(self)))

      def producerConnected(producer: ActorRef) = {
        case msg => producer ! msg
      }
    }))
  }

  /**
   * Declare a queue asynchronously.
   * When the broker has a connection to the channel, the queue will be declared.
   *
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the queue name, pass None to generate one randomly
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @param clientFacing true if the queue will be read by an external client
   * @return the future with the queue name
   */
  def declareQueue(exchangeName: String, routingKey: String, queueName: Option[String] = None,
    handlerActor: Option[ActorRef] = None, clientFacing: Boolean = false): Future[String] = {

    val p = Promise[String]()
    val finalQueueName =
      queueName.getOrElse(if (clientFacing) generateClientFacingName() else generateName())

    val extraArgs: Map[String, AnyRef] =
      if (clientFacing) Map("x-message-ttl" -> Long.box(clientFacingMessageTTL), "x-expires" -> Long.box(clientFacingQueueTTL))
      else Map.empty

    val init = Seq(
      DeclareQueue(QueueParameters(finalQueueName, passive = false, args = extraArgs)),
      QueueBind(finalQueueName, exchange = exchangeName, routing_key = routingKey))

    createChannel(channel => {

      val proxy = rmqActorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case ok @ Ok(_: DeclareQueue, res: Option[_]) =>
            context become receiveBind(res.get.asInstanceOf[Queue.DeclareOk].getQueue)
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
        def receiveBind(queueName: String): Receive = {
          case ok @ Ok(_: QueueBind, res: Option[_]) =>
            p trySuccess queueName
            handlerActor.foreach(channel ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
      }))

      init.foreach(request => channel.tell(Record(request), proxy.actorRef))
    })
    p.future
  }

  /**
   * Bind a specific queue to a given actor asynchronously.
   * When the broker has a connection to the channel, the queue is declared and bound to the given actor.
   *
   * @param actor the actor which will listen to the queue
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the queue name, pass None to generate one randomly
   * @param autodelete if true, the queue will be destroyed when it is no longer used
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @param clientFacing true if the queue will be read by an external client
   * @return the future with the consumer actor and the queue name
   */
  def setListeningConsumer(actor: ActorRef, exchangeName: String, routingKey: String, queueName: Option[String] = None,
    autodelete: Boolean = true, handlerActor: Option[ActorRef] = None,
    clientFacing: Boolean = false): Future[(ActorRef, String)] = {
    val p = Promise[(ActorRef, String)]()

    val finalQueueName =
      queueName.getOrElse(if (clientFacing) generateClientFacingName() else generateName())

    val extraArgs: Map[String, AnyRef] =
      if (clientFacing) Map("x-message-ttl" -> Long.box(clientFacingMessageTTL), "x-expires" -> Long.box(clientFacingQueueTTL))
      else Map.empty

    val init = Seq(
      DeclareQueue(QueueParameters(finalQueueName, passive = false, autodelete = autodelete, args = extraArgs)),
      QueueBind(finalQueueName, exchange = exchangeName, routing_key = routingKey),
      AddQueue(QueueParameters(finalQueueName, passive = false, autodelete = autodelete)))

    createConsumer(actor, consumer => {

      val proxy = rmqActorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case ok @ Ok(_: DeclareQueue, res: Option[_]) =>
            context become receiveBind(res.get.asInstanceOf[Queue.DeclareOk].getQueue)
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
        def receiveBind(queueName: String): Receive = {
          case ok @ Ok(_: QueueBind, res: Option[_]) =>
            p trySuccess (consumer, queueName)
            handlerActor.foreach(consumer ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
      }))

      init.foreach(request => consumer.tell(Record(request), proxy.actorRef))
    })
    p.future
  }

  /**
   * Bind a specific queue to a given actor asynchronously.
   * When the broker has a connection to the channel, the queue is declared and bound to the given actor.
   *
   * @param actor the actor which will listen to the queue
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the queue name, pass None to generate one randomly
   * @param autodelete if true, the queue will be destroyed when it is no longer used
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @param clientFacing true if the queue will be read by an external client
   * @return the future with the queue name
   */
  def setListeningQueue(actor: ActorRef, exchangeName: String, routingKey: String, queueName: Option[String] = None,
    autodelete: Boolean = true, handlerActor: Option[ActorRef] = None,
    clientFacing: Boolean = false): Future[String] = {
    implicit val ec = rmqActorRefFactory.dispatcher
    setListeningConsumer(actor, exchangeName, routingKey, queueName, autodelete, handlerActor, clientFacing).map(_._2)
  }

  /**
   * Setup a RpcServer asynchronously.
   * When the broker has a connection to the channel, the exchange and queues are declared and then it will start
   * to process incoming messages.
   *
   * The queue name can be None to create a One-To-Many (all servers/processors handle the request) or specified
   * for a One-To-Any (only one server/processor will handle the request).
   *
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the queue name
   * @param server the function that processes an Delivery
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *   [[com.github.sstone.amqp.ChannelOwner.Disconnected]], [[com.github.sstone.amqp.ChannelOwner.Connected]] or a
   *   [[com.github.sstone.amqp.Amqp.Error]].
   * @param timeout an optional message timeout to avoid filling up the queue with old, already ignored, responses
   * @param ec the implicit execution context
   * @return the future with the queue name
   */
  def createRpcServer(exchangeName: String, routingKey: String, queueName: Option[String] = None,
    handlerActor: Option[ActorRef] = None, timeout: Option[FiniteDuration] = None)(server: Delivery => Future[ProcessResult])(implicit ec: ExecutionContext) = {

    val finalQueueName = queueName.getOrElse(generateName())

    val extraArgs: Map[String, AnyRef] = timeout match {
      case Some(t) => Map("x-message-ttl" -> Long.box(t.toMillis))
      case _ => Map.empty
    }

    val p = Promise[String]()

    declareExchange(exchangeName, handlerActor = handlerActor).foreach(_ => {

      val processor = new IProcessor {
        def process(delivery: Delivery) = server(delivery)
        def onFailure(delivery: Delivery, e: Throwable) = ProcessResult(None)
      }

      val exchangeParams = ExchangeParameters(exchangeName, passive = false, "topic", durable = true)
      val queueParams = QueueParameters(finalQueueName, passive = false, args = extraArgs)

      val rpcServerProps = RpcServer.props(processor, channelParams = Some(ChannelParameters(1)))

      rpcActor(rpcServerProps, rpcChannel => {
        handlerActor.foreach(rpcChannel ! AddStatusListener(_))
        val proxy = rmqActorRefFactory.actorOf(Props(new Actor {
          def receive = {
            case ok @ Ok(_: AddBinding, _) =>
              p trySuccess finalQueueName
            case error: Error =>
              handlerActor.foreach(_ forward error)
              p tryFailure error.reason
          }
        }))

        rpcChannel.tell(Record(AddBinding(Binding(exchangeParams, queueParams, routingKey))), proxy.actorRef)
      })

    })

    p.future
  }

  /**
   * Create a rpc client
   *
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   * @return the rpc client actor ref
   */
  def createRpcClient(handlerActor: Option[ActorRef] = None): Future[ActorRef] = {
    val p = Promise[ActorRef]()

    rpcActor(RpcClient.props(), rpcChannel => {

      val proxy = rmqActorRefFactory.actorOf(Props(new Actor {
        def receive = {

          case com.github.sstone.amqp.ChannelOwner.Connected =>
            p trySuccess rpcChannel

          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason

        }
      }))

      rpcChannel ! AddStatusListener(proxy)
    })

    p.future
  }
}
