package eu.shiftforward.adstax.ups.api

import scala.concurrent.Future

import akka.stream.scaladsl.{ Sink, Source }
import akka.{ Done, NotUsed }
import io.circe._
import org.joda.time.DateTime

/**
 * Interface for clients capable of interacting with the User Profile Storage module by retrieving, updating or deleting
 * information about users.
 */
trait UserProfileStorageClient {

  /**
   * Retrieves all attributes of a user.
   *
   * @param userId the identifier of the user
   * @return a `Future` with the user attributes wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def get(userId: UserId): Future[Option[UserData]]

  /**
   * Returns all ids associated with the provided UserId. The provided UserId is always included in the set.
   *
   * @param userId the identifier of the user whose set of ids we want to fetch
   * @return a `Future` with the set of users this UserId belongs to.
   */
  def getIds(userId: UserId): Future[Set[UserId]]

  /**
   * Retrieves all ids associated with the provided UserId, along with its attributes.
   *
   * @param userId the identifier of the user whose set of ids and attributes we want to fetch
   * @return a `Future` containing a tuple with:
   *         - the set of UserIds this UserId belongs to
   *         - user attributes wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def getIdsAndData(userId: UserId): Future[(Set[UserId], Option[UserData])]

  /**
   * Retrieves a single base attribute of a user.
   *
   * @param userId the identifier of the user
   * @param name the name of the base attribute
   * @return a `Future` with the attribute value wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def getBaseAttribute(userId: UserId, name: String): Future[Option[Json]]

  /**
   * Retrieves a single computed attribute of a user.
   *
   * @param userId the identifier of the user
   * @param name the name of the computed attribute
   * @return a `Future` with the attribute value wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def getComputedAttribute(userId: UserId, name: String): Future[Option[Json]]

  /**
   * Updates the attributes of a user.
   *
   * @param userId the identifier of the user
   * @param attributes the attributes to update the user with
   * @param mergeStrategy the strategy to use when merging the attributes
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def update(
    userId: UserId,
    attributes: UserAttributes,
    mergeStrategy: AttributesMergingStrategy = MergeMergingStrategy): Future[Unit]

  /**
   * Deletes the information about a user.
   *
   * @param userId the identifier of the user
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def delete(userId: UserId): Future[Unit]

  /**
   * Deletes an attribute of a user.
   * @param userId the identifier of the userId
   * @param attributeName the name of the attribute
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def delete(userId: UserId, attributeName: String): Future[Unit]

  /**
   * Links two users. After linked, the users will be unified, so that their attributes will be shared and getting,
   * updating, and deleting attributes works interchangeably between them. The provided source can later be used to
   * reset the created links.
   *
   * @param userId1 the identifier of the first user
   * @param userId2 the identifier of the second user
   * @param source an optional source to use for the link. Links with named sources can later be reset
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def link(userId1: UserId, userId2: UserId, source: Option[String] = None): Future[Unit]

  /**
   * Sets the merge strategy for a single base attribute. If an attribute doesn't have a merging strategy defined, it is
   * assumed to be [[MostRecent]].
   *
   * @param name the name of the attribute whose merging strategy we would like to define
   * @param mergeStrategy the strategy to use from now onwards when merging attributes of the provided name
   * @return a `Future` that is completed when the operation is acknowledged.
   */
  def setMergeStrategy(name: String, mergeStrategy: SingleAttributeMergingStrategy): Future[Unit]

  /**
   * Resets all links of the provided source.
   *
   * @param source the source of the links to be reset
   * @return a `Future` that is completed when the operation is acknowledged.
   */
  def resetLinks(source: String): Future[Unit]

  /**
   * A [[akka.stream.scaladsl.Sink]] for user update operations.
   */
  def updateSink: Sink[UpdateUser, Future[Done]]

  /**
   * A [[akka.stream.scaladsl.Source]] of users of a given type along with their attributes.
   *
   * @param idType the type of identifier
   * @return the source of user information
   */
  def userSource(idType: String): Source[(UserId, UserData), NotUsed]

  /**
   * A [[akka.stream.scaladsl.Source]] of users of a given type that were updated during the range of supplied
   * `DateTime`s, along with their attributes.
   *
   * @param idType the type of identifier
   * @param from the lower bound of the time range
   * @param to the upper bound of the time range
   * @return the source of user information
   */
  def userSource(idType: String, from: DateTime, to: DateTime): Source[(UserId, UserData), NotUsed]
}
