package eu.shiftforward.adstax.ups.api

import io.circe._
import io.circe.generic.semiauto._
import org.joda.time.DateTime

import com.velocidi.apso.json.ExtraJsonProtocol._

/**
 * A request to be sent to the User Profile Storage module.
 */
sealed trait UserProfileStorageRequest

/**
 * A request to retrieve all attributes of a user.
 *
 * @param userId the identifier of the user
 */
case class GetUser(userId: UserId) extends UserProfileStorageRequest

/**
 * A request to fetch the set of ids associated with a given user
 *
 * @param userId the id of the user to fetch the set of ids for
 */
case class GetIds(userId: UserId) extends UserProfileStorageRequest

/**
 * A request to retrieve the set of ids associated with a given user along with its attributes.
 *
 * @param userId the id of the user to fetch the set of ids and attributes for
 */
case class GetIdsAndData(userId: UserId) extends UserProfileStorageRequest

/**
 * A request to retrieve a single base attribute of a user.
 *
 * @param userId the identifier of the user
 * @param name the name of the base attribute
 */
case class GetUserBaseAttribute(userId: UserId, name: String) extends UserProfileStorageRequest

/**
 * A request to retrieve a single computed attribute of a user.
 *
 * @param userId the identifier of the user
 * @param name the name of the computed attribute
 */
case class GetUserComputedAttribute(userId: UserId, name: String) extends UserProfileStorageRequest

/**
 * A request to update the information about a user.
 *
 * @param userId the identifier of the user
 * @param attributes the attributes to update the user with
 * @param mergeStrategy the strategy to use when merging the attributes
 * @param timestamp the timestamp of the update
 */
case class UpdateUser(
    userId: UserId,
    attributes: UserAttributes,
    mergeStrategy: AttributesMergingStrategy,
    timestamp: DateTime = DateTime.now) extends UserProfileStorageRequest

object UpdateUser {
  implicit val updateUserJsonEncoder: Encoder[UpdateUser] = deriveEncoder[UpdateUser]
  implicit val updateUserJsonDecoder: Decoder[UpdateUser] = deriveDecoder[UpdateUser]
}

/**
 * A request to delete the information about a user.
 *
 * @param userId the identifier of the user
 */
case class DeleteUser(userId: UserId) extends UserProfileStorageRequest

/**
 * A request to delete an attribute of a user.
 *
 * @param userId the identifier of the user
 * @param attributeName the name of the attribute
 */
case class DeleteUserAttribute(userId: UserId, attributeName: String) extends UserProfileStorageRequest

/**
 * A request to link the two users.
 *
 * @param userId1 the identifier of the first user
 * @param userId2 the identifier of the second user
 * @param source an optional source for the link
 */
case class LinkUsers(userId1: UserId, userId2: UserId, source: Option[String]) extends UserProfileStorageRequest

/**
 * A request to set the merge strategy of a given base attribute.
 *
 * @param name the name of the base attribute whose merge strategy is to be set
 * @param mergeStrategy the merge strategy to set for the attribute
 */
case class SetMergeStrategy(name: String, mergeStrategy: SingleAttributeMergingStrategy) extends UserProfileStorageRequest

/**
 * A request to reset the links of a given source.
 *
 * @param source the source to be reset
 */
case class ResetLinks(source: String) extends UserProfileStorageRequest
