package eu.shiftforward.adstax.ups.api

import scala.concurrent.duration._

import io.circe._
import io.circe.syntax._

import com.velocidi.apso.json.ExtraJsonProtocol._

/**
 * A strategy to merge two attributes.
 */
sealed trait SingleAttributeMergingStrategy

/**
 * A basic strategy to merge two attributes, not taking any period into account.
 */
sealed trait BasicSingleAttributeMergingStrategy extends SingleAttributeMergingStrategy

/**
 * A strategy to merge two attributes in the provided period.
 */
case class PeriodicSingleAttributeMergingStrategy(period: FiniteDuration, strategy: BasicSingleAttributeMergingStrategy)
  extends SingleAttributeMergingStrategy

/**
 * Attribute merging strategy that prefers the attribute value of the most recently changed user when merging
 * attributes.
 */
case object MostRecent extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that prefers the attribute value of the least recently changed user when merging
 * attributes.
 */
case object Older extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that sums both attribute values when merging attributes. When merging two attributes,
 * attribute values that are not numbers nor can't be directly converted to one are considered to be numbers with value
 * 0.
 *
 * Strings whose value can be parsed to a double can be converted to a number.
 * Booleans with value `false` can be converted to a number with value 0.
 * Booleans with value `true` can be converted to a number with value 1.
 */
case object Sum extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that uses the max of both attribute values as the final one. When merging two attributes,
 * attribute values that are not numbers nor can't be directly converted to one are considered to be numbers with value
 * `Double.MinValue`.
 *
 * Strings whose value can be parsed to a double can be converted to a number.
 * Booleans with value `false` can be converted to a number with value 0.
 * Booleans with value `true` can be converted to a number with value 1.
 */
case object Max extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that uses the min of both attribute values as the final one. When merging two attributes,
 * attribute values that are not numbers nor can't be directly converted to one are considered to be numbers with value
 * `Double.MaxValue`.
 *
 * Strings whose value can be parsed to a double can be converted to a number.
 * Booleans with value `false` can be converted to a number with value 0.
 * Booleans with value `true` can be converted to a number with value 1.
 */
case object Min extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that joins all the individual elements of both attribute values in a single array. When
 * merging two attributes, attribute values that are not arrays are converted to an array  whose value is a sequence
 * with a single element: the value of the attribute prior to the conversion.
 */
case object Concatenate extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that merges the attribute values, provided that they are both maps. When conflicting keys
 * arise, this merging strategy preferes the value of the most recently changed user. When merging two attributes,
 * attribute values that are not maps are considered to be an empty map.
 */
case object MapMerge extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that performs a boolean AND between both attribute values. When merging two attributes,
 * attribute values that are not booleans nor can't be directly converted to one are considered to be booleans with
 * value `true`.
 *
 * Numbers with value different from `0` are converted to booleans with value `true`.
 * Numbers with value equal to `0` are converted to booleans with value `false`.
 * Strings with value `"true"` are converted to booleans with value `true`.
 * Strings with value `"false"` are converted to booleans with value `false`.
 * Strings whose value can be parsed to a Double follow the rules of the number conversion enumerated above.
 */
case object And extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that performs a boolean OR between both attribute values. When merging two attributes,
 * attribute values that are not booleans nor can't be directly converted to one are considered to be booleans with
 * value `false`.
 *
 * Numbers with value different from `0` are converted to booleans with value `true`.
 * Numbers with value equal to `0` are converted to booleans with value `false`.
 * Strings with value `"true"` are converted to booleans with value `true`.
 * Strings with value `"false"` are converted to booleans with value `false`.
 * Strings whose value can be parsed to a Double follow the rules of the number conversion enumerated above.
 */
case object Or extends BasicSingleAttributeMergingStrategy

object BasicSingleAttributeMergingStrategy {
  private val fromStr = Map[String, BasicSingleAttributeMergingStrategy](
    "most-recent" -> MostRecent,
    "older" -> Older,
    "sum" -> Sum,
    "max" -> Max,
    "min" -> Min,
    "concatenate" -> Concatenate,
    "map-merge" -> MapMerge,
    "and" -> And,
    "or" -> Or)
  private val toStr = fromStr.map { case (k, v) => v -> k }

  implicit val basicSingleAttributeStrategyJsonEncoder: Encoder[BasicSingleAttributeMergingStrategy] =
    Encoder[String].contramap(toStr)

  implicit val basicSingleAttributeStrategyJsonDecoder: Decoder[BasicSingleAttributeMergingStrategy] =
    Decoder[String].emap(s =>
      fromStr.get(s) match {
        case Some(v) => Right(v)
        case None => Left(s"Invalid single attribute merging strategy: $s.")
      })
}

object PeriodicSingleAttributeMergingStrategy {
  private val strToTimeUnit = Map(
    "d" -> DAYS,
    "h" -> HOURS,
    "μs" -> MICROSECONDS,
    "ms" -> MILLISECONDS,
    "m" -> MINUTES,
    "ns" -> NANOSECONDS,
    "s" -> SECONDS)
  private val timeUnitToStr = strToTimeUnit.map { case (k, v) => v -> k }

  private val periodicSingleAttributeStrategyR = "([^\\|]+)\\|(\\d+.+)".r

  implicit val periodicSingleAttributeStrategyJsonEncoder: Encoder[PeriodicSingleAttributeMergingStrategy] =
    Encoder[String].contramap {
      case PeriodicSingleAttributeMergingStrategy(ps, ms) =>
        s"${ms.asJson.asString.get}|${ps.length}${timeUnitToStr(ps.unit)}"
    }

  implicit val perodicSingleAttributeStrategyJsonDecoder: Decoder[PeriodicSingleAttributeMergingStrategy] = {
    val sDecoder = Decoder[BasicSingleAttributeMergingStrategy]
    val dDecoder = Decoder[FiniteDuration]

    Decoder[String].emap {
      case periodicSingleAttributeStrategyR(str, duration) =>
        sDecoder.decodeJson(str.asJson).right.flatMap { m =>
          dDecoder.decodeJson(duration.asJson).right.map { d =>
            PeriodicSingleAttributeMergingStrategy(d, m)
          }
        }.left.map(_.message)
      case other =>
        Left(s"Invalid single attribute merging strategy: $other.")
    }
  }
}

object SingleAttributeMergingStrategy {
  implicit val singleAttributeStrategyJsonEncoder: Encoder[SingleAttributeMergingStrategy] =
    Encoder.instance({
      case v: BasicSingleAttributeMergingStrategy => v.asJson
      case v: PeriodicSingleAttributeMergingStrategy => v.asJson
    })

  implicit val singleAttributeStrategyJsonDecoder: Decoder[SingleAttributeMergingStrategy] =
    Decoder[BasicSingleAttributeMergingStrategy] or
      (Decoder[PeriodicSingleAttributeMergingStrategy] or
        Decoder.failedWithMessage[SingleAttributeMergingStrategy]("Invalid single attribute merging strategy"))
}
