package eu.shiftforward.adstax.ups.api

import io.circe._

/**
 * A strategy to merge two sets of user attributes.
 */
sealed trait AttributesMergingStrategy

/**
 * Simple attributes merge strategy that updates the old attributes with the most recent ones. Attributes that were only
 * present in the old attributes are preserved. [[SingleAttributeMergingStrategy]]s are applied for individual
 * attributes when using this attributes merge strategy.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (b, 1), (c, 1)]
 */
case object MergeMergingStrategy extends AttributesMergingStrategy

/**
 * Simple attribute merge strategy that overwrites the old attributes with new attributes, removing all old attributes.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (c, 1)]
 */
case object ReplaceMergingStrategy extends AttributesMergingStrategy

object AttributesMergingStrategy {
  implicit val mergingStrategyJsonEncoder: Encoder[AttributesMergingStrategy] = Encoder[String].contramap({
    case MergeMergingStrategy => "merge"
    case ReplaceMergingStrategy => "replace"
  })

  implicit val mergingStrategyJsonDecoder: Decoder[AttributesMergingStrategy] = Decoder[String].emap {
    case "merge" => Right(MergeMergingStrategy)
    case "replace" => Right(ReplaceMergingStrategy)
    case strategy => Left(s"Invalid attribute merging strategy: $strategy.")
  }
}
