package eu.shiftforward.adstax.tracking

import akka.actor._
import com.github.sstone.amqp.Amqp.Delivery
import io.circe.parser
import pureconfig.syntax._

import com.velocidi.apso.config.LazyConfigFactory
import eu.shiftforward.adstax.config
import eu.shiftforward.adstax.util.AmqpClient

/**
 * Base trait containing functionality for consuming tracking events from an AdStax Event Tracker instance. The
 * processing step is defined by a `Props` which creates worker actors able to process JSON events. The trait also
 * allows configuring the number of workers to instantiate, providing support for parallel consuming of events.
 */
trait TrackingListener {

  /**
   * The AMQP client to use to receive tracking events.
   */
  def amqp: AmqpClient

  /**
   * The `ActorRef` factory, such as an `ActorSystem` or `ActorContext`, used to create the actors needed for tracking.
   */
  def actorRefFactory: ActorRefFactory

  /**
   * The tracking configuration.
   */
  def trackingConfig: config.RabbitMQ.Binding

  /**
   * A `Props` encapsulating the creation of worker actors. The actor should be able to receive and process JSON events
   * in the form of [[io.circe.Json]] objects.
   */
  def processorProps: Props

  /**
   * The number of worker actors to instantiate. The instantiated workers will process events in parallel.
   */
  def numProcessors: Int

  implicit private[this] def ec = amqp.dispatcher

  private[this] class JsonDeserializerActor(queueName: String) extends Actor {
    val processor = context.actorOf(processorProps)
    amqp.addConsumer(self, trackingConfig.exchangeName, trackingConfig.routingKey, Some(queueName))

    def receive = {
      case Delivery(_, _, _, body) =>
        parser.parse(new String(body, "UTF-8")).right.foreach { json =>
          processor ! json
        }
    }
  }

  amqp.declareQueue(trackingConfig.exchangeName, trackingConfig.routingKey).foreach { queueName =>
    for (_ <- 1 to numProcessors)
      actorRefFactory.actorOf(Props(new JsonDeserializerActor(queueName)))
  }
}

/**
 * An actor able to consume tracking events from an AdStax Event Tracker instance.
 * @param processorProps a `Props` encapsulating the creation of worker actors
 * @param numProcessors the number of worker actors to instantiate
 * @param amqpFactory a function building a AMQP client from an `ActorRefFactory`
 * @param trackingConfig the tracking configuration
 */
class TrackingListenerActor(
    val processorProps: Props,
    val numProcessors: Int,
    val amqpFactory: ActorRefFactory => AmqpClient,
    val trackingConfig: config.RabbitMQ.Binding) extends Actor with TrackingListener {

  def this(processorProps: Props, numProcessors: Int, amqp: AmqpClient, trackingConfig: config.RabbitMQ.Binding) =
    this(processorProps, numProcessors, _ => amqp, trackingConfig)

  lazy val amqp = amqpFactory(context)
  def actorRefFactory = context

  def receive = Actor.emptyBehavior
}

/**
 * Companion object containing factory methods for `TrackingListenerActor` actors.
 */
object TrackingListenerActor {

  /**
   * Creates a `Props` for creating `TrackingListenerActor` actors with the default settings.
   *
   * @param processorProps a `Props` encapsulating the creation of worker actors
   * @param numProcessors the number of worker actors to instantiate
   * @param amqpConfig the AMQP configuration
   * @param httpEventsRmqBinding the rmq binding configuration for http events
   * @return a `Props` for creating `TrackingListenerActor` actors.
   */
  def props(
    processorProps: Props,
    numProcessors: Int = 1,
    amqpConfig: config.RabbitMQ = LazyConfigFactory.load.getConfig("adstax.sdk.event-tracker.rabbitmq").toOrThrow[config.RabbitMQ],
    httpEventsRmqBinding: config.RabbitMQ.Binding = LazyConfigFactory.load.getConfig("adstax.sdk.event-tracker.rmq-bindings.http-events").toOrThrow[config.RabbitMQ.Binding]) = {

    Props(new TrackingListenerActor(
      processorProps,
      numProcessors,
      { refFact => new AmqpClient(amqpConfig, "tracking-listener")(refFact) },
      httpEventsRmqBinding))
  }

  /**
   * Creates a `Props` for creating `TrackingListenerActor` actors with an already created `AmqpClient`.
   *
   * @param processorProps a `Props` encapsulating the creation of worker actors
   * @param numProcessors the number of worker actors to instantiate
   * @param amqp the AMQP client
   * @param trackingConfig the tracking configuration
   * @return a `Props` for creating `TrackingListenerActor` actors.
   */
  def props(processorProps: Props, numProcessors: Int, amqp: AmqpClient, trackingConfig: config.RabbitMQ.Binding) =
    Props(new TrackingListenerActor(processorProps, numProcessors, amqp, trackingConfig))
}
