package eu.shiftforward.adstax.util.rpc

import scala.concurrent.Future
import scala.concurrent.duration._

import akka.actor.ActorRefFactory
import akka.pattern.ask
import akka.util.Timeout
import com.github.sstone.amqp.Amqp.Publish
import com.github.sstone.amqp.RpcClient.Request
import com.github.sstone.amqp.{ Amqp, RpcClient }
import spray.json._

import eu.shiftforward.adstax.config
import eu.shiftforward.adstax.util.AmqpClient

/**
 * A generic client for RPC clients using AMQP as the transport layer.
 */
trait AmqpRpcJsonClient {

  /**
   * The AMQP client to use
   */
  def amqp: AmqpClient

  /**
   * The AMQP RPC config
   */
  def rpcConfig: config.RpcClient

  implicit lazy val ec = amqp.dispatcher
  implicit val timeout = Timeout(10.seconds)

  private[this] val clientExchangeName = rpcConfig.exchangeName
  private[this] val clientRoutingKeyPrefix = rpcConfig.routingKeyPrefix

  private[this] val clientActor = amqp.declareExchange(clientExchangeName).flatMap(_ => amqp.createRpcClient())

  /**
   * Dispatches a request message with a Future of a concrete response type.
   *
   * @param request the request message
   * @tparam Req the type of the request message
   * @tparam Resp the response type that the callback returns
   * @return a Future of the expected response message
   */
  def dispatchRequest[Req, Resp](request: Req)(
    implicit
    td: TypeDescriptor[Req, Resp], writer: JsonWriter[Req], reader: JsonFormat[Resp]): Future[Resp] = {

    dispatch(request, td.routingKey) { response =>
      new String(response.body, "UTF-8").parseJson.convertTo[RpcResponse[Resp]]
    }
  }

  private[this] def dispatch[Req: JsonWriter, Resp](req: Req, routingKey: String)(
    f: Amqp.Delivery => RpcResponse[Resp]): Future[Resp] = {

    clientActor.flatMap { client =>
      (client ? Request(Publish(clientExchangeName, clientRoutingKeyPrefix + routingKey, req.toJson.compactPrint.getBytes("UTF-8")))).flatMap {
        case response: RpcClient.Response =>
          f(response.deliveries.head) match {
            case RpcError(msg) =>
              Future.failed(new Exception("[AmqpRpcJsonClient] Error: " + msg))
            case RpcSuccess(res) =>
              Future.successful(res)
          }
        case error: RpcClient.Undelivered =>
          Future.failed(new Exception("[AmqpRpcJsonClient] Undelivered Message: " + error.msg))
      }
    }
  }
}

object AmqpRpcJsonClient {

  /**
   * Creates a `AmqpRpcJsonClient` sing an AMQP client and a RPC config.
   *
   * @param _amqp the AMQP client to use
   * @param _rpcConfig the AMQP RPC config
   * @return a `AmqpRpcJsonClient` uing the given AMQP client and RPC config
   */
  def apply(_amqp: AmqpClient, _rpcConfig: config.RpcClient): AmqpRpcJsonClient = new AmqpRpcJsonClient {
    lazy val amqp = _amqp
    lazy val rpcConfig = _rpcConfig
  }

  /**
   * Creates a `AmqpRpcJsonClient` sing an client and a RPC config.
   *
   * @param amqpConfig the AMQP config
   * @param rpcConfig the AMQP RPC config
   * @return a `AmqpRpcJsonClient` using the given AMQP config and RPC config
   */
  def apply(amqpConfig: config.RabbitMQ, rpcConfig: config.RpcClient)(implicit actorRefFactory: ActorRefFactory): AmqpRpcJsonClient =
    apply(new AmqpClient(amqpConfig, "adstax-sdk"), rpcConfig)
}
