package eu.shiftforward.adstax.ups.api

import scala.concurrent.duration._

import spray.json.DefaultJsonProtocol._
import spray.json._

/**
 * A strategy to merge two attributes.
 */
sealed trait SingleAttributeMergingStrategy

/**
 * A basic strategy to merge two attributes, not taking any period into account.
 */
sealed trait BasicSingleAttributeMergingStrategy extends SingleAttributeMergingStrategy

/**
 * A strategy to merge two attributes in the provided period.
 */
case class PeriodicSingleAttributeMergingStrategy(period: FiniteDuration, strategy: BasicSingleAttributeMergingStrategy)
  extends SingleAttributeMergingStrategy

/**
 * Attribute merging strategy that prefers the attribute value of the most recently changed user when merging
 * attributes.
 */
case object MostRecent extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that prefers the attribute value of the least recently changed user when merging
 * attributes.
 */
case object Older extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that sums both attribute values when merging attributes. When merging two attributes,
 * attribute values that are not [[NumericAttrValue]]s nor can't be directly converted to one are considered to be
 * [[NumericAttrValue]]s with value 0.
 *
 * [[StringAttrValue]]s whose value can be parsed to a double can be converted to a [[NumericAttrValue]].
 * [[BooleanAttrValue]]s with value `false` can be converted to a [[NumericAttrValue]] with value 0.
 * [[BooleanAttrValue]]s with value `true` can be converted to a [[NumericAttrValue]] with value 1.
 */
case object Sum extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that uses the max of both attribute values as the final one. When merging two attributes,
 * attribute values that are not [[NumericAttrValue]]s nor can't be directly converted to one are considered to be
 * [[NumericAttrValue]]s with value `Double.MinValue`.
 *
 * [[StringAttrValue]]s whose value can be parsed to a double can be converted to a [[NumericAttrValue]].
 * [[BooleanAttrValue]]s with value `false` can be converted to a [[NumericAttrValue]] with value 0.
 * [[BooleanAttrValue]]s with value `true` can be converted to a [[NumericAttrValue]] with value 1.
 */
case object Max extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that uses the min of both attribute values as the final one. When merging two attributes,
 * attribute values that are not [[NumericAttrValue]]s nor can't be directly converted to one are considered to be
 * [[NumericAttrValue]]s with value `Double.MaxValue`.
 *
 * [[StringAttrValue]]s whose value can be parsed to a double can be converted to a [[NumericAttrValue]].
 * [[BooleanAttrValue]]s with value `false` can be converted to a [[NumericAttrValue]] with value 0.
 * [[BooleanAttrValue]]s with value `true` can be converted to a [[NumericAttrValue]] with value 1.
 */
case object Min extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that joins all the individual elements of both attribute values in a single array. When
 * merging two attributes, attribute values that are not [[ArrayAttrValue]]s are converted to an [[ArrayAttrValue]]
 * whose value is a sequence with a single element: the value of [[AttributeValue]] prior to the conversion.
 */
case object Concatenate extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that merges the attribute values, provided that they are both maps. When conflicting keys
 * arise, this merging strategy preferes the value of the most recently changed user. When merging two attributes,
 * attribute values that are not [[MapAttrValue]]s are considered to be [[MapAttrValue]] with a value of `Map.empty`.
 */
case object MapMerge extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that performs a boolean AND between both attribute values. When merging two attributes,
 * attribute values that are not [[BooleanAttrValue]]s nor can't be directly converted to one are considered to be
 * [[BooleanAttrValue]]s with value `true`.
 *
 * [[NumericAttrValue]]s with value different from `0` are converted to [[BooleanAttrValue]]s with value `true`.
 * [[NumericAttrValue]]s with value equal to `0` are converted to [[BooleanAttrValue]]s with value `false`.
 * [[StringAttrValue]]s with value `"true"` are converted to [[BooleanAttrValue]]s with value `true`.
 * [[StringAttrValue]]s with value `"false"` are converted to [[BooleanAttrValue]]s with value `false`.
 * [[StringAttrValue]]s whose value can be parsed to a Double follow the rules of the [[NumericAttrValue]] conversion
 * enumerated above.
 */
case object And extends BasicSingleAttributeMergingStrategy

/**
 * Attribute merging strategy that performs a boolean OR between both attribute values. When merging two attributes,
 * attribute values that are not [[BooleanAttrValue]]s nor can't be directly converted to one are considered to be
 * [[BooleanAttrValue]]s with value `false`.
 *
 * [[NumericAttrValue]]s with value different from `0` are converted to [[BooleanAttrValue]]s with value `true`.
 * [[NumericAttrValue]]s with value equal to `0` are converted to [[BooleanAttrValue]]s with value `false`.
 * [[StringAttrValue]]s with value `"true"` are converted to [[BooleanAttrValue]]s with value `true`.
 * [[StringAttrValue]]s with value `"false"` are converted to [[BooleanAttrValue]]s with value `false`.
 * [[StringAttrValue]]s whose value can be parsed to a Double follow the rules of the [[NumericAttrValue]] conversion
 * enumerated above.
 */
case object Or extends BasicSingleAttributeMergingStrategy

object SingleAttributeMergingStrategy {
  private val strToBasicMergeStrategy = Map[String, BasicSingleAttributeMergingStrategy](
    "most-recent" -> MostRecent,
    "older" -> Older,
    "sum" -> Sum,
    "max" -> Max,
    "min" -> Min,
    "concatenate" -> Concatenate,
    "map-merge" -> MapMerge,
    "and" -> And,
    "or" -> Or)
  private val basicMergeStrategyToStr = strToBasicMergeStrategy.map { case (k, v) => v -> k }

  private val strToTimeUnit = Map(
    "d" -> DAYS,
    "h" -> HOURS,
    "μs" -> MICROSECONDS,
    "ms" -> MILLISECONDS,
    "m" -> MINUTES,
    "ns" -> NANOSECONDS,
    "s" -> SECONDS)
  private val timeUnitToStr = strToTimeUnit.map { case (k, v) => v -> k }

  private val periodicSingleAttributeStrategyR = "([^\\|]+)\\|(\\d+)(.+)".r

  implicit object SingleAttributeMergingStrategyJsonFormat extends JsonFormat[SingleAttributeMergingStrategy] {
    def read(json: JsValue) = json match {
      case JsString(str) if strToBasicMergeStrategy.contains(str) =>
        strToBasicMergeStrategy(str)

      case JsString(periodicSingleAttributeStrategyR(str, length, unit)) if strToBasicMergeStrategy.contains(str) && strToTimeUnit.contains(unit) =>
        PeriodicSingleAttributeMergingStrategy(FiniteDuration(length.toLong, strToTimeUnit(unit)), strToBasicMergeStrategy(str))

      case other => deserializationError(s"Invalid single attribute merging strategy: ${other.compactPrint}")
    }

    def write(obj: SingleAttributeMergingStrategy) = obj match {
      case b: BasicSingleAttributeMergingStrategy =>
        basicMergeStrategyToStr(b).toJson
      case PeriodicSingleAttributeMergingStrategy(ps, ms) =>
        s"${basicMergeStrategyToStr(ms)}|${ps.length}${timeUnitToStr(ps.unit)}".toJson
    }
  }
}
