package eu.shiftforward.adstax.scheduler.api

import scala.concurrent.Future

import eu.shiftforward.adstax.scheduler.api.job.SchedulerJob

/**
 * Interface for clients capable of interacting with the Scheduler module for scheduling new jobs and managing existing
 * ones.
 */
trait SchedulerClient {

  /**
   * Schedules a given action in the scheduler.
   *
   * @param action the action to be scheduled
   * @return a `Future` with the action just scheduled.
   */
  def scheduleAction(action: SchedulerJob): Future[JobStatus]

  /**
   * Cancels a previously scheduled action.
   *
   * @param id the id of the action to cancel
   * @return a `Future ` with the id of the action just cancelled.
   */
  def cancelAction(id: String): Future[JobStatus]

  /**
   * Returns the status of all previously scheduled jobs.
   *
   * @return a `Future` with the Map of all the Scheduled jobs, indexed by job id.
   */
  def getJobsStatus: Future[Map[String, JobStatus]]

  /**
   * Returns the status of the provided scheduler action.
   *
   * @param id the id of the scheduled action
   * @return a `Future` with the current status of the scheduled action, `None` if the action doesn't exist.
   */
  def getJobStatus(id: String): Future[Option[JobStatus]]
}
