package eu.shiftforward.adstax.ups.api

import scala.concurrent.Future

import akka.stream.scaladsl.{ Sink, Source }
import akka.{ Done, NotUsed }
import org.joda.time.DateTime

/**
 * Interface for clients capable of interacting with the User Profile Storage module by retrieving, updating or deleting
 * information about users.
 */
trait UserProfileStorageClient {

  /**
   * Retrieves the attributes of an user.
   *
   * @param userId the identifier of the user
   * @return a `Future` with the user attributes wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def get(userId: UserId): Future[Option[UserAttributes]]

  /**
   * Retrieves a single attribute of an user.
   *
   * @param userId the identifier of the user
   * @param attributeName the name of the attribute
   * @return a `Future` with the attribute value wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def get(userId: UserId, attributeName: String): Future[Option[AttributeValue]]

  /**
   * Updates the attributes of an user.
   *
   * @param userId the identifier of the user
   * @param attributes the attributes to update the user with
   * @param mergeStrategy the strategy to use when merging the attributes
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def update(
    userId: UserId,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy = JoinMergingStrategy): Future[Unit]

  /**
   * Deletes the information about a user.
   *
   * @param userId the identifier of the user
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def delete(userId: UserId): Future[Unit]

  /**
   * Deletes an attribute of a user.
   * @param userId the identifier of the userId
   * @param attributeName the name of the attribute
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def delete(userId: UserId, attributeName: String): Future[Unit]

  /**
   * Links two users. After linked, the users will be unified, so that their attributes will be shared and getting,
   * updating, and deleting attributes works interchangeably between them. The provided source can later be used to
   * reset the created links.
   *
   * @param userId1 the identifier of the first user
   * @param userId2 the identifier of the second user
   * @param source an optional source to use for the link. Links with named sources can later be reset
   * @return a `Future ` that is completed when the operation is acknowledged.
   */
  def link(userId1: UserId, userId2: UserId, source: Option[String] = None): Future[Unit]

  /**
   * Resets all links of the provided source.
   *
   * @param source the source of the links to be reset
   * @return a `Future` that is completed when the operation is acknowledged.
   */
  def resetLinks(source: String): Future[Unit]

  /**
   * A [[akka.stream.scaladsl.Sink]] for user update operations.
   */
  def updateSink: Sink[UpdateUser, Future[Done]]

  /**
   * A [[akka.stream.scaladsl.Source]] of users of a given type along with their information.
   *
   * @param idType the type of identifier
   * @return the source of user information
   */
  def userSource(idType: String): Source[(UserId, UserAttributes), NotUsed]

  /**
   * A [[akka.stream.scaladsl.Source]] of users of a given type that were updated during the range of supplied
   * `DateTime`s.
   *
   * @param idType the type of identifier
   * @param from the initial `DateTime` to consider for the last user update
   * @param to the final `DateTime` to consider for the last user update
   * @return the source of user information
   */
  def userSource(idType: String, from: DateTime, to: DateTime): Source[(UserId, UserAttributes), NotUsed]
}
