package eu.shiftforward.adstax.ups.api

import spray.json.DefaultJsonProtocol._
import spray.json._

import eu.shiftforward.apso.Implicits._

/**
 * The value of an attribute. An attribute takes one of a fixed set of attribute types.
 */
sealed trait AttributeValue {

  /**
   * The value of this attribute as a native Scala value
   */
  def value: Any

  /**
   * Merges this attribute with another. If both attributes are of collection types a proper merge is performed, with
   * child attributes of the the argument having precedence over the attributes of this one. If one of them is scalar,
   * the argument is returned.
   *
   * @param that the attribute to merge with this one
   * @return the merged attribute.
   */
  def merge(that: AttributeValue): AttributeValue = (this, that) match {
    case (MapAttrValue(thisMap), MapAttrValue(thatMap)) =>
      MapAttrValue(thisMap.twoWayMerge(thatMap)((js1, js2) => js1.merge(js2)))
    case (ArrayAttrValue(thisSeq), ArrayAttrValue(thatSeq)) => ArrayAttrValue(thisSeq ++ thatSeq)
    case _ => that
  }
}

/**
 * A string attribute value.
 *
 * @param value the string value
 */
case class StringAttrValue(value: String) extends AttributeValue

/**
 * A numeric attribute value.
 *
 * @param value the number value
 */
case class NumericAttrValue(value: Double) extends AttributeValue

/**
 * A boolean attribute value.
 *
 * @param value the boolean value
 */
case class BooleanAttrValue(value: Boolean) extends AttributeValue

/**
 * An attribute value representing a set of lan-lon coordinates.
 *
 * @param lat the latitude value
 * @param lon the longitude value
 */
case class GeoAttrValue(lat: Double, lon: Double) extends AttributeValue {
  def value = (lat, lon)
}

/**
 * An array attribute value.
 *
 * @param arr the array of attribute values
 */
case class ArrayAttrValue(arr: Seq[AttributeValue]) extends AttributeValue {
  def value: Seq[Any] = arr.map(_.value)
}

/**
 * A map attribute value.
 *
 * @param map the map of keys to attribute values
 */
case class MapAttrValue(map: Map[String, AttributeValue]) extends AttributeValue {
  def value = map.mapValues(_.value)
}

object AttributeValue {

  implicit val attrValueFormat = new JsonFormat[AttributeValue] {

    def read(json: JsValue): AttributeValue = json match {
      case JsString(x) => StringAttrValue(x)
      case JsNumber(x) => NumericAttrValue(x.toDouble)
      case JsBoolean(x) => BooleanAttrValue(x)
      case JsArray(x) => ArrayAttrValue(x.map(read))
      case JsObject(x) if x.keySet == Set("lat", "long") =>
        GeoAttrValue(x("lat").convertTo[Double], x("long").convertTo[Double])
      case JsObject(x) => MapAttrValue(x.mapValues(read))
    }

    def write(obj: AttributeValue): JsValue = obj match {
      case StringAttrValue(str) => str.toJson
      case NumericAttrValue(x) => x.toJson
      case BooleanAttrValue(b) => b.toJson
      case ArrayAttrValue(arr) => arr.map(write).toJson
      case GeoAttrValue(lat, lon) => JsObject("lat" -> lat.toJson, "lon" -> lon.toJson)
      case MapAttrValue(map) => map.mapValues(write).toJson
    }
  }

  object Conversions {
    implicit def stringAttributeWrap(str: String): AttributeValue = StringAttrValue(str)
    implicit def numericAttributeWrap[T](num: T)(implicit n: Numeric[T]): AttributeValue = NumericAttrValue(n.toDouble(num))
    implicit def booleanAttributeWrap(bool: Boolean): AttributeValue = BooleanAttrValue(bool)
    implicit def arrayAttributeWrap[T](arr: Array[T])(implicit w: T => AttributeValue): AttributeValue = ArrayAttrValue(arr.map(w))
    implicit def arrayAttributeWrap[T](arr: Seq[T])(implicit w: T => AttributeValue): AttributeValue = ArrayAttrValue(arr.map(w))
    implicit def mapAttributeWrap[T](map: Map[String, T])(implicit w: T => AttributeValue): AttributeValue = MapAttrValue(map.mapValues(w))
    implicit def mapAttributeWrap(map: Map[String, AttributeValue]): AttributeValue = MapAttrValue(map)

    implicit def stringAttributeUnwrap(attr: StringAttrValue): String = attr.value
    implicit def numericAttributeUnwrap(attr: NumericAttrValue): Double = attr.value
    implicit def booleanAttributeUnwrap(attr: BooleanAttrValue): Boolean = attr.value
    implicit def arrayAttributeUnwrap(attr: ArrayAttrValue): Seq[Any] = attr.value
    implicit def mapAttributeUnwrap(attr: MapAttrValue): Map[String, Any] = attr.value
  }
}
