package eu.shiftforward.adstax.ups.api

import spray.json.DefaultJsonProtocol._

/**
 * A request to be sent to the User Profile Storage module.
 */
sealed trait UserProfileStorageRequest

/**
 * A request to retrieve information about a user.
 *
 * @param userId the identifier of the user
 */
case class GetUser(userId: UserId) extends UserProfileStorageRequest

object GetUser {
  implicit val getFormat = jsonFormat1(GetUser.apply)
}

/**
 * A request to retrieve a single attribute of a user.
 *
 * @param userId the identifier of the user
 * @param attributeName the name of the attribute
 */
case class GetUserAttribute(userId: UserId, attributeName: String) extends UserProfileStorageRequest

object GetUserAttribute {
  implicit val getAttributeFormat = jsonFormat2(GetUserAttribute.apply)
}

/**
 * A request to update the information about a user.
 *
 * @param userId the identifier of the user
 * @param attributes the attributes to update the user with
 * @param mergeStrategy the strategy to use when merging the attributes
 */
case class UpdateUser(
  userId: UserId,
  attributes: UserAttributes,
  mergeStrategy: AttributeMergingStrategy) extends UserProfileStorageRequest

object UpdateUser {
  implicit val updateFormat = jsonFormat3(UpdateUser.apply)
}

/**
 * A request to delete the information about a user.
 *
 * @param userId the identifier of the user
 */
case class DeleteUser(userId: UserId) extends UserProfileStorageRequest

object DeleteUser {
  implicit val deleteFormat = jsonFormat1(DeleteUser.apply)
}

/**
 * A request to delete an attribute of a user.
 *
 * @param userId the identifier of the user
 * @param attributeName the name of the attribute
 */
case class DeleteUserAttribute(userId: UserId, attributeName: String) extends UserProfileStorageRequest

object DeleteUserAttribute {
  implicit val deleteAttributeFormat = jsonFormat2(DeleteUserAttribute.apply)
}

/**
 * A request to merge the information about two users.
 *
 * @param userId1 the identifier of the first user
 * @param userId2 the identifier of the second user
 */
case class MergeUsers(userId1: UserId, userId2: UserId) extends UserProfileStorageRequest

object MergeUsers {
  implicit val mergeFormat = jsonFormat2(MergeUsers.apply)
}
