package eu.shiftforward.adstax.ups.api

import io.circe._

import eu.shiftforward.apso.Implicits._

/**
 * The attribute set of a user.
 *
 * @param attributes the map from attribute names to values
 */
case class UserAttributes(attributes: Map[String, Json]) {
  private[this] def attrToValue(json: Json): Any =
    json.fold(
      null,
      identity,
      _.toDouble,
      identity,
      _.map(attrToValue),
      _.toMap.mapValues(attrToValue).map(identity))

  /**
   * Converts this attribute set to a native Scala collection.
   *
   * @return a native Scala collection containing the attribute set.
   */
  def toSymbolTable: Map[String, Any] =
    attributes.mapValues(attrToValue)

  /**
   * Merges this attribute set with another. The merge is performed attribute by attribute using their `merge` method,
   * with the attribute set passed as argument having precedence over this one.
   *
   * @param that the attribute set to merge with this one
   * @return the merged attribute set.
   */
  def merge(that: UserAttributes): UserAttributes =
    UserAttributes(this.attributes.twoWayMerge(that.attributes)(_.deepMerge(_)))

  /**
   * Returns the attribute value associated with the provided name.
   *
   * @param name the name of the attribute whose value is to be returned
   * @return a `Some` with the value of the attribute with the provided name if it exists, `None` otherwise.
   */
  def get(name: String): Option[Json] =
    attributes.get(name)
}

object UserAttributes {

  /**
   * The empty set of user attributes.
   */
  val empty = UserAttributes(Map.empty[String, Json])

  implicit val userAttributesJsonEncoder: Encoder[UserAttributes] =
    Encoder[Map[String, Json]].contramap[UserAttributes](_.attributes)
  implicit val userAttributesJsonDecoder: Decoder[UserAttributes] =
    Decoder[Map[String, Json]].map(UserAttributes.apply)
}
