package eu.shiftforward.adstax.ups.api

import spray.json.DefaultJsonProtocol._
import spray.json._

import eu.shiftforward.apso.Implicits._
import eu.shiftforward.apso.json.Implicits._

/**
 * The attribute set of a user.
 *
 * @param attributes the map from attribute names to values
 */
case class UserAttributes(attributes: Map[String, JsValue]) {

  // this differs from the `toValue` method in `eu.shiftforward.apso.json.Implicits` only on the behavior with
  // `JsNumber`, where this method converts it to a `Double`
  private[this] def attrToValue(json: JsValue): Any = json match {
    case JsString(str) => str
    case JsNumber(num) => num.toDouble
    case JsObject(map) => map.mapValues(attrToValue).map(identity)
    case JsArray(elems) => elems.map(attrToValue)
    case JsBoolean(bool) => bool
    case JsNull => null
  }

  /**
   * Converts this attribute set to a native Scala collection.
   *
   * @return a native Scala collection containing the attribute set.
   */
  def toSymbolTable: Map[String, Any] =
    attributes.mapValues(attrToValue)

  /**
   * Merges this attribute set with another. The merge is performed attribute by attribute using their `merge` method,
   * with the attribute set passed as argument having precedence over this one.
   *
   * @param that the attribute set to merge with this one
   * @return the merged attribute set.
   */
  def merge(that: UserAttributes): UserAttributes =
    UserAttributes(this.attributes.twoWayMerge(that.attributes)(_.merge(_, false)))

  /**
   * Returns the attribute value associated with the provided name.
   *
   * @param name the name of the attribute whose value is to be returned
   * @return a `Some` with the value of the attribute with the provided name if it exists, `None` otherwise.
   */
  def get(name: String): Option[JsValue] =
    attributes.get(name)
}

object UserAttributes {

  /**
   * The empty set of user attributes.
   */
  val empty = UserAttributes(Map.empty[String, JsValue])

  implicit object UserAttributesJsonFormat extends RootJsonFormat[UserAttributes] {
    def write(ua: UserAttributes) = ua.attributes.toJson
    def read(value: JsValue) = UserAttributes(value.convertTo[Map[String, JsValue]])
  }
}
