package eu.shiftforward.adstax.ups.api

import spray.json._

/**
 * A strategy to merge two sets of user attributes.
 */
sealed trait AttributesMergingStrategy

/**
 * Simple attributes merge strategy that updates the old attributes with the most recent ones. Attributes that were only
 * present in the old attributes are preserved. [[SingleAttributeMergingStrategy]]s are applied for individual
 * attributes when using this attributes merge strategy.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (b, 1), (c, 1)]
 */
case object MergeMergingStrategy extends AttributesMergingStrategy

/**
 * Simple attribute merge strategy that overwrites the old attributes with new attributes, removing all old attributes.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (c, 1)]
 */
case object ReplaceMergingStrategy extends AttributesMergingStrategy

object AttributesMergingStrategy {
  implicit object AttributesMergingStrategyJsonFormat extends JsonFormat[AttributesMergingStrategy] {
    def read(json: JsValue) = json match {
      case JsString("merge") => MergeMergingStrategy
      case JsString("replace") => ReplaceMergingStrategy
      case strategy => deserializationError(s"Invalid attribute merging strategy: ${strategy.compactPrint}")
    }

    def write(obj: AttributesMergingStrategy) = obj match {
      case MergeMergingStrategy => JsString("merge")
      case ReplaceMergingStrategy => JsString("replace")
    }
  }
}
