package eu.shiftforward.adstax.productfeeder.api

import scala.xml._

import org.apache.commons.lang3.StringEscapeUtils
import org.joda.time.DateTime
import spray.json.DefaultJsonProtocol._
import spray.json._

import eu.shiftforward.apso.json.ExtraJsonProtocol._
import eu.shiftforward.apso.json.Implicits._
import eu.shiftforward.apso.json.JsonConvert

/**
 * Represents a product.
 *
 * @param id its unique id
 * @param attributes a map of attributes
 * @param lastUpdated a timestamp for when this product was last updated
 */
case class ProductItem(id: String, attributes: Map[String, Any], lastUpdated: DateTime = DateTime.now())

object ProductItem {

  object JsonProtocol {
    implicit val mapAnyJsonFormat = new RootJsonFormat[Map[String, Any]] {
      def write(m: Map[String, Any]) = JsonConvert.toJson(m)
      def read(json: JsValue) = json.toValue.asInstanceOf[Map[String, Any]]
    }
  }

  /**
   * The ProductItem's JsonProtocol object, with conversions to and from JSON.
   */
  implicit val productItemJsonFormat: RootJsonFormat[ProductItem] = {
    import JsonProtocol._
    jsonFormat3(ProductItem.apply)
  }

  object XmlProtocol {
    /**
     * Takes an XML document and converts it to a List of Maps
     */
    def mapNodeToAnyFormat(node: Elem): List[Map[String, Any]] = {

      // maps the given 'from' root node label, to a list of all values from the matched nodes with the same, given label
      def getNodesAsValueList(from: Node, nodeLabel: String): Map[String, Any] = {
        val listValues = Map(nodeLabel -> (from \\ nodeLabel).toList.map { n =>
          getNodeAsValueMap(n)(nodeLabel)
        })
        listValues.filter {
          case (_, v) => v.nonEmpty
        }
      }

      def getNodeAsValueMap(node: Node): Map[String, Any] = {
        if (node.text.trim.isEmpty) {
          Map.empty
        } else if (node.child.count(_.isInstanceOf[Text]) == 1) {
          Map(node.label -> StringEscapeUtils.unescapeXml(node.text.trim))
        } else {

          val nodeLabels = node.child.map(_.label)
          val repeatedLabels = nodeLabels.diff(nodeLabels.distinct).distinct

          val listValues = repeatedLabels.map(getNodesAsValueList(node, _))

          val simpleValues = node.child.foldLeft(Map.empty[String, Any]) { (acc, elem) =>
            if (!repeatedLabels.contains(elem.label))
              acc ++ getNodeAsValueMap(elem)
            else acc
          }

          Map(node.label -> (listValues.fold(Map.empty)(_ ++ _) ++ simpleValues))
        }
      }

      // assumes "entry" or "item" nodes, as they are the nodes that should exist in RSS or Atom formats, respectively
      val productsNodes = {
        val productsNodesAtomFormat = node \\ "entry"

        if (productsNodesAtomFormat.nonEmpty) productsNodesAtomFormat
        else node \\ "item"
      }

      productsNodes.map { xmlNode =>
        val productMap = getNodeAsValueMap(xmlNode)("entry").asInstanceOf[Map[String, Any]]

        Map(
          "id" -> productMap("id"),
          "attributes" -> (productMap - "id"))
      }.toList
    }
  }
}
