package eu.shiftforward.adstax.ups.api

import spray.json.DefaultJsonProtocol._

/**
 * A request to be sent to the User Profile Storage module.
 */
sealed trait UserProfileStorageRequest

/**
 * A request to retrieve all attributes of a user.
 *
 * @param userId the identifier of the user
 */
case class GetUser(userId: UserId) extends UserProfileStorageRequest

object GetUser {
  implicit val getFormat = jsonFormat1(GetUser.apply)
}

/**
 * A request to retrieve a single base attribute of a user.
 *
 * @param userId the identifier of the user
 * @param name the name of the base attribute
 */
case class GetUserBaseAttribute(userId: UserId, name: String) extends UserProfileStorageRequest

object GetUserBaseAttribute {
  implicit val getBaseAttributeFormat = jsonFormat2(GetUserBaseAttribute.apply)
}

/**
 * A request to retrieve a single computed attribute of a user.
 *
 * @param userId the identifier of the user
 * @param name the name of the computed attribute
 */
case class GetUserComputedAttribute(userId: UserId, name: String) extends UserProfileStorageRequest

object GetUserComputedAttribute {
  implicit val getComputedAttributeFormat = jsonFormat2(GetUserComputedAttribute.apply)
}

/**
 * A request to update the information about a user.
 *
 * @param userId the identifier of the user
 * @param attributes the attributes to update the user with
 * @param mergeStrategy the strategy to use when merging the attributes
 */
case class UpdateUser(
  userId: UserId,
  attributes: UserAttributes,
  mergeStrategy: AttributeMergingStrategy) extends UserProfileStorageRequest

object UpdateUser {
  implicit val updateFormat = jsonFormat3(UpdateUser.apply)
}

/**
 * A request to delete the information about a user.
 *
 * @param userId the identifier of the user
 */
case class DeleteUser(userId: UserId) extends UserProfileStorageRequest

object DeleteUser {
  implicit val deleteFormat = jsonFormat1(DeleteUser.apply)
}

/**
 * A request to delete an attribute of a user.
 *
 * @param userId the identifier of the user
 * @param attributeName the name of the attribute
 */
case class DeleteUserAttribute(userId: UserId, attributeName: String) extends UserProfileStorageRequest

object DeleteUserAttribute {
  implicit val deleteAttributeFormat = jsonFormat2(DeleteUserAttribute.apply)
}

/**
 * A request to link the two users.
 *
 * @param userId1 the identifier of the first user
 * @param userId2 the identifier of the second user
 * @param source an optional source for the link
 */
case class LinkUsers(userId1: UserId, userId2: UserId, source: Option[String]) extends UserProfileStorageRequest

object LinkUsers {
  implicit val linkFormat = jsonFormat3(LinkUsers.apply)
}

/**
 * A request to fetch the set of ids associated with a given user
 *
 * @param userId the id of the user to fetch the set of ids for
 */
case class GetIds(userId: UserId) extends UserProfileStorageRequest

object GetIds {
  implicit val getIdsFormat = jsonFormat1(GetIds.apply)
}

/**
 * A request to reset the links of a given source.
 *
 * @param source the source to be reset
 */
case class ResetLinks(source: String) extends UserProfileStorageRequest

object ResetLinks {
  implicit val resetFormat = jsonFormat1(ResetLinks.apply)
}
