package eu.shiftforward.adstax.ups.api.rpc

import akka.actor.ActorRefFactory
import org.joda.time.DateTime
import pureconfig.syntax._

import eu.shiftforward.adstax.config
import eu.shiftforward.adstax.ups.api._
import eu.shiftforward.adstax.ups.api.rpc.TypeDescriptors._
import eu.shiftforward.adstax.util.AmqpClient
import eu.shiftforward.adstax.util.rpc.AmqpRpcJsonClient
import eu.shiftforward.apso.config.LazyConfigFactory

/**
 * The AMQP RPC Client to interact with the User Profile Storage.
 */
class UserProfileStorageAmqpRpcClient(val amqp: AmqpClient, val rpcConfig: config.RpcClient)
    extends UserProfileStorageClient with AmqpRpcJsonClient {

  def get(userId: UserId) =
    dispatchRequest(GetUser(userId)).map(_.attributes)

  def get(userId: UserId, attributeName: String) =
    dispatchRequest(GetUserAttribute(userId, attributeName)).map(_.attributeValue)

  def update(
    userId: UserId,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy = JoinMergingStrategy) =
    dispatchRequest(UpdateUser(userId, attributes, mergeStrategy)).map(_ => ())

  def delete(userId: UserId) =
    dispatchRequest(DeleteUser(userId)).map(_ => ())

  def delete(userId: UserId, attributeName: String) =
    dispatchRequest(DeleteUserAttribute(userId, attributeName)).map(_ => ())

  def link(userId1: UserId, userId2: UserId, source: Option[String] = None) =
    dispatchRequest(LinkUsers(userId1, userId2, source)).map(_ => ())

  def getIds(userId: UserId) =
    dispatchRequest(GetIds(userId)).map(_.ids)

  def resetLinks(source: String) =
    dispatchRequest(ResetLinks(source)).map(_ => ())

  def updateSink =
    throw new UnsupportedOperationException("Streaming operations are not supported via AMQP RPC")

  def userSource(idType: String) =
    throw new UnsupportedOperationException("Streaming operations are not supported via AMQP RPC")

  def userSource(idType: String, from: DateTime, to: DateTime) =
    throw new UnsupportedOperationException("Streaming operations are not supported via AMQP RPC")
}

object UserProfileStorageAmqpRpcClient {

  def apply(amqp: AmqpClient, rpcConfig: config.RpcClient): UserProfileStorageAmqpRpcClient =
    new UserProfileStorageAmqpRpcClient(amqp, rpcConfig)

  def apply(
    amqpConfig: config.RabbitMQ = LazyConfigFactory.load.getConfig("adstax.sdk.user-profile-storage.rabbitmq").to[config.RabbitMQ].get,
    rpcConfig: config.RpcClient = LazyConfigFactory.load.getConfig("adstax.sdk.user-profile-storage.rpc").to[config.RpcClient].get)(
    implicit
    actorRefFactory: ActorRefFactory): UserProfileStorageAmqpRpcClient = {

    new UserProfileStorageAmqpRpcClient(new AmqpClient(amqpConfig, "adstax-user-profile-storage-sdk"), rpcConfig)
  }
}
