package eu.shiftforward.adstax.util.rpc

import spray.json.DefaultJsonProtocol._
import spray.json._

/**
 * A response to a RPC call.
 */
sealed trait RpcResponse[+T]

/**
 * An error occurred as a result of a RPC call.
 *
 * @param message the error message
 */
case class RpcError(message: String) extends RpcResponse[Nothing]

object RpcError {
  implicit val errorJsonFormat: RootJsonFormat[RpcError] = new RootJsonFormat[RpcError] {
    def read(json: JsValue) = json match {
      case JsObject(fields) if fields.contains("message") =>
        RpcError(fields("message").convertTo[String])
      case other => deserializationError(s"Expected a JsObject with field 'message', got: $other")
    }

    def write(obj: RpcError) = JsObject(
      "type" -> JsString("RpcError"),
      "message" -> obj.message.toJson)
  }
}

/**
 * A successful RPC response.
 *
 * @param response the RPC response
 */
case class RpcSuccess[T](response: T) extends RpcResponse[T]

object RpcSuccess {
  implicit def successJsonFormat[T: JsonFormat]: RootJsonFormat[RpcSuccess[T]] = new RootJsonFormat[RpcSuccess[T]] {
    def read(json: JsValue) = json match {
      case JsObject(fields) if fields.contains("response") =>
        RpcSuccess(fields("response").convertTo[T])
      case other => deserializationError(s"Expected a JsObject with field 'response', got: $other")
    }

    def write(obj: RpcSuccess[T]) = JsObject(
      "type" -> JsString("RpcSuccess"),
      "response" -> obj.response.toJson)
  }
}

object RpcResponse {
  implicit def responseJsonFormat[T: JsonFormat]: RootJsonFormat[RpcResponse[T]] = new RootJsonFormat[RpcResponse[T]] {
    def read(json: JsValue) = json match {
      case JsObject(fields) if fields.contains("type") =>
        fields("type") match {
          case JsString("RpcError") => json.convertTo[RpcError]
          case JsString("RpcSuccess") => json.convertTo[RpcSuccess[T]]
          case other => deserializationError(s"Unknown RPC response type $other")
        }
      case other => deserializationError(s"Expected a JsObject with field 'type', got: $other")
    }

    def write(obj: RpcResponse[T]) = obj match {
      case error: RpcError => error.toJson
      case success: RpcSuccess[T] => success.toJson
    }
  }
}
