package eu.shiftforward.adstax.productfeeder.api

import scala.concurrent.Future

/**
 * Interface for clients capable of interacting with the Product Feeder module, by fetching, updating or deleting
 * products.
 */
trait ProductFeederClient {

  /**
   * Retrieves a product.
   *
   * @param clientId the identifier of the client
   * @param siteId the identifier of the site
   * @param productId the identifier of the product
   * @return a `Future` with the [[ProductItem]] wrapped in a `Some` if it exists, `None` otherwise.
   */
  def getProduct(clientId: String, siteId: String, productId: String): Future[Option[ProductItem]]

  /**
   * Updates the product details.
   *
   * @param clientId the identifier of the client
   * @param siteId the optional identifier of the site. If set to `null` the product will be updated at a client level;
   *               otherwise it will be updated at the site level.
   * @param productId the identifier of the product
   * @param productAttributes a map of keys to product attributes to use for updating the current product definition
   * @return a `Future ` with `true` if the product update was successful, `false` otherwise.
   */
  def updateProduct(clientId: String, siteId: String, productId: String, productAttributes: Map[String, Any]): Future[Boolean]

  /**
   * Deletes the product details.
   *
   * @param clientId the identifier of the client
   * @param siteId the optional identifier of the site. If set to `null` the product will be deleted at a client level;
   *               otherwise it will be deleted at the site level.
   * @param productId the identifier of the product
   * @return a `Future` with `true` if the product delete was successful, `false` otherwise.
   */
  def deleteProduct(clientId: String, siteId: String, productId: String): Future[Boolean]
}
