package eu.shiftforward.adstax.scheduler.api.job

import scala.concurrent.duration._

import shapeless._

import eu.shiftforward.adstax.scheduler.api.RecurrentScript
import eu.shiftforward.adstax.scheduler.api.job.SchedulerJob.RepetitionConfig

/**
 * A job that publishes messages to a AMQP broker.
 *
 * @param id the identifier of the job
 * @param initialDelay the initial delay before the job is first executed
 * @param repeat the repetition configuration for the job, or `None` if it's an one-shot task
 * @param messageDefinition the script to be run to build each message. The script should return an object with fields
 *                          `exchangeKey`, `routingKey` and `message`.
 */
case class AmqpSchedulerJob(
    id: String,
    initialDelay: FiniteDuration,
    repeat: Option[RepetitionConfig],
    messageDefinition: RecurrentScript) extends SchedulerJob {

  val action = AmqpSchedulerJob.Action
}

object AmqpSchedulerJob extends SchedulerJobCompanion[AmqpSchedulerJob] {
  final val Action = "publish-to-amqp"

  object JsonProtocol {
    val formatBuilder = SchedulerJob.JsonProtocol.baseFormatBuilder
      .field[RecurrentScript]("messageDefinition")
  }

  implicit lazy val jobJsonFormat = JsonProtocol.formatBuilder.jsonFormat[AmqpSchedulerJob](
    {
      case id :: `Action` :: initialDelay :: repeat :: messageDefinition :: HNil =>
        AmqpSchedulerJob(id, initialDelay, repeat, messageDefinition)
    },
    { job => job.id :: Action :: job.initialDelay :: job.repeat :: job.messageDefinition :: HNil })
}
