package eu.shiftforward.adstax.util

import java.util.UUID
import java.util.concurrent.TimeoutException

import scala.concurrent.duration._
import scala.concurrent.{ Future, Promise }

import akka.actor._
import com.github.sstone.amqp.Amqp._
import com.github.sstone.amqp.RpcServer.{ IProcessor, ProcessResult }
import com.github.sstone.amqp._
import com.rabbitmq.client.AMQP.Queue
import com.rabbitmq.client.ConnectionFactory

import eu.shiftforward.adstax.config

/**
 * An AMQP client simplifying some of the common interactions with AMQP brokers. It handles correctly cases in which
 * the broker is not reachable at the time operations are done, waiting for a connection to be established first.
 *
 * @param amqpConfig the AMQP broker config
 * @param queueNamespace a namespace which all queues created without a name will have
 */
class AmqpClient(amqpConfig: config.RabbitMQ, queueNamespace: String = "")(implicit actorRefFactory: ActorRefFactory) {

  /**
   * Generates a name for an anonymous queue.
   * This function should generate ids that make debug easier, while having an
   * easily identifiable name to allow custom policies.
   * Required since the suggested official implementation (ie. "") does not work properly.
   *
   * @return a random queue name
   */
  private[this] def generateName() = s"gen-$queueNamespace-${UUID.randomUUID()}"

  /**
   * Generates a name for an anonymous client-facing queue.
   * This function should generate clean ids that can be shown to a client, while having an
   * easily identifiable name to allow custom policies.
   * Required since the suggested official implementation (ie. "") does not work properly.
   *
   * @return a random queue name
   */
  private[this] def generateClientFacingName() = s"out-${UUID.randomUUID()}"

  private[this] lazy val waitForAmqpTimeout = amqpConfig.timeout
  private[this] lazy val clientFacingMessageTTL = amqpConfig.cfMessageTtl.get.toMillis
  private[this] lazy val clientFacingQueueTTL = amqpConfig.cfQueueTtl.get.toMillis

  lazy val connectionOwner = {
    val amqpHost = amqpConfig.host
    val amqpPort = amqpConfig.port
    val amqpUser = amqpConfig.username
    val amqpPass = amqpConfig.password

    val connFactory = new ConnectionFactory()
    connFactory.setHost(amqpHost)
    connFactory.setPort(amqpPort)
    connFactory.setUsername(amqpUser)
    connFactory.setPassword(amqpPass)

    actorRefFactory.actorOf(ConnectionOwner.props(connFactory))
  }

  private[this] def createChildActor(props: Props) = try {
    ConnectionOwner.createChildActor(conn = connectionOwner, channelOwner = props, timeout = waitForAmqpTimeout)
  } catch {
    case _: TimeoutException => throw new TimeoutException("Could not connect to RabbitMQ")
  }

  private[this] def createChannel(onConnected: ActorRef => Unit): ActorRef = {
    val channel = createChildActor(ChannelOwner.props())

    Amqp.onConnection(actorRefFactory, channel, () => onConnected(channel))
    channel
  }

  private[this] def createConsumer(actor: ActorRef, onConnected: ActorRef => Unit): ActorRef = {
    val consumer = createChildActor(Consumer.props(
      listener = actor,
      autoack = true,
      channelParams = Some(ChannelParameters(1))))

    Amqp.onConnection(actorRefFactory, consumer, () => onConnected(consumer))
    consumer
  }

  private[this] def rpcActor(props: Props, onConnected: ActorRef => Unit): ActorRef = {
    val rpcActor = createChildActor(props)

    Amqp.onConnection(actorRefFactory, rpcActor, () => onConnected(rpcActor))
    rpcActor
  }

  /**
   * The `ExecutionContext` used by this client to schedule asynchronous tasks.
   */
  implicit def dispatcher = actorRefFactory.dispatcher

  /**
   * Declares an exchange.
   *
   * @param exchangeName the exchange name
   * @param exchangeType the exchange type
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @return a `Future` that is completed when the exchange is successfully declared.
   */
  def declareExchange(
    exchangeName: String,
    exchangeType: String = "topic",
    handlerActor: Option[ActorRef] = None): Future[Unit] = {

    val p = Promise[Unit]()

    val declareExchange =
      DeclareExchange(ExchangeParameters(exchangeName, passive = false, exchangeType, durable = true))

    createChannel(channel => {

      val proxy = actorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case _@ Ok(_: DeclareExchange, _) =>
            p.trySuccess(())
            handlerActor.foreach(channel ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p.tryFailure(error.reason)
        }
      }))

      channel.tell(Record(declareExchange), proxy)
    })
    p.future
  }

  /**
   * Creates a producer actor. Producer actors can be sent [[com.github.sstone.amqp.Amqp.Publish]], which they dispatch
   * to the AMQP broker. The created actor will discard publish messages that are sent at a time in which the client is
   * disconnected from the broker.
   *
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @return a future that is completed when the exchange is successfully declared.
   * @return the `ActorRef` of the producer.
   */
  def createProducer(handlerActor: Option[ActorRef]): ActorRef = {
    val producer = createChildActor(ChannelOwner.props())

    Amqp.onConnection(actorRefFactory, producer, () => {
      handlerActor.foreach(producer ! AddStatusListener(_))
    })

    producer
  }

  /**
   * Creates a producer actor which stashes messages until the producer is fully connected to the AMQP broker. Producer
   * actors can be sent [[com.github.sstone.amqp.Amqp.Publish]], which they dispatch to the AMQP broker.
   *
   * @return the `ActorRef` of the producer.
   */
  def createStashedProducer(): ActorRef = {
    actorRefFactory.actorOf(Props(new BaseAmqpProducerStashActor {
      def setupAmqp = Future.successful(createProducer(Some(self)))

      def producerConnected(producer: ActorRef) = {
        case msg => producer ! msg
      }
    }))
  }

  /**
   * Declares a queue and binds it to an exchange and routing key.
   *
   * @param exchangeName the exchange name to bind the queue to
   * @param routingKey the routing key to bind the queue to
   * @param queueName the queue name wrapped in a `Some`, or `None` to generate one with a random suffix
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @param clientFacing `true` if the queue will be read by an AdStax-external client, `false` otherwise
   * @return a `Future` that is completed with the queue name when the queue is successfully bound.
   */
  def declareQueue(
    exchangeName: String,
    routingKey: String,
    queueName: Option[String] = None,
    handlerActor: Option[ActorRef] = None,
    clientFacing: Boolean = false): Future[String] = {

    val p = Promise[String]()
    val finalQueueName =
      queueName.getOrElse(if (clientFacing) generateClientFacingName() else generateName())

    val extraArgs: Map[String, AnyRef] =
      if (clientFacing) Map("x-message-ttl" -> Long.box(clientFacingMessageTTL), "x-expires" -> Long.box(clientFacingQueueTTL))
      else Map.empty

    val init = Seq(
      DeclareQueue(QueueParameters(finalQueueName, passive = false, args = extraArgs)),
      QueueBind(finalQueueName, exchange = exchangeName, routing_key = routingKey))

    createChannel(channel => {

      val proxy = actorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case _@ Ok(_: DeclareQueue, res: Option[_]) =>
            context become receiveBind(res.get.asInstanceOf[Queue.DeclareOk].getQueue)
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
        def receiveBind(queueName: String): Receive = {
          case _@ Ok(_: QueueBind, _: Option[_]) =>
            p trySuccess queueName
            handlerActor.foreach(channel ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
      }))

      init.foreach(request => channel.tell(Record(request), proxy))
    })
    p.future
  }

  /**
   * Declares a queue, binds it to an exchange and routing key and adds a consumer to it.
   *
   * @param actor the actor which will consume [[com.github.sstone.amqp.Amqp.Delivery]] messages
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the underlying queue name wrapped in a `Some`, or `None` to generate one with a random suffix
   * @param autodelete `true` if the queue should be destroyed when it is no longer used, `false` otherwise
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @param clientFacing `true` if the queue will be read by an AdStax-external client, `false` otherwise
   * @return a `Future` that is completed with the queue name when the consumer is successfully added.
   */
  def addConsumer(
    actor: ActorRef,
    exchangeName: String,
    routingKey: String,
    queueName: Option[String] = None,
    autodelete: Boolean = true, handlerActor: Option[ActorRef] = None,
    clientFacing: Boolean = false): Future[String] = {

    addConsumerWithBindingActor(actor, exchangeName, routingKey, queueName, autodelete, handlerActor, clientFacing)
      .map(_._2)
  }

  /**
   * Declares a queue, binds it to an exchange and routing key and adds a consumer to it. Returns additionally the
   * underlying actor doing the binding, which can be killed later to stop the consumption of AMQP messages.
   *
   * @param actor the actor which will consume [[com.github.sstone.amqp.Amqp.Delivery]] messages
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the underlying queue name wrapped in a `Some`, or `None` to generate one with a random suffix
   * @param autodelete `true` if the queue should be destroyed when it is no longer used, `false` otherwise
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @param clientFacing `true` if the queue will be read by an AdStax-external client, `false` otherwise
   * @return a `Future` that is completed with the underlying binding actor and the queue name when the consumer is
   *         successfully added.
   */
  def addConsumerWithBindingActor(
    actor: ActorRef,
    exchangeName: String,
    routingKey: String,
    queueName: Option[String] = None,
    autodelete: Boolean = true,
    handlerActor: Option[ActorRef] = None,
    clientFacing: Boolean = false): Future[(ActorRef, String)] = {

    val p = Promise[(ActorRef, String)]()

    val finalQueueName =
      queueName.getOrElse(if (clientFacing) generateClientFacingName() else generateName())

    val extraArgs: Map[String, AnyRef] =
      if (clientFacing) Map("x-message-ttl" -> Long.box(clientFacingMessageTTL), "x-expires" -> Long.box(clientFacingQueueTTL))
      else Map.empty

    val init = Seq(
      DeclareQueue(QueueParameters(finalQueueName, passive = false, autodelete = autodelete, args = extraArgs)),
      QueueBind(finalQueueName, exchange = exchangeName, routing_key = routingKey),
      AddQueue(QueueParameters(finalQueueName, passive = false, autodelete = autodelete)))

    createConsumer(actor, consumer => {

      val proxy = actorRefFactory.actorOf(Props(new Actor {
        def receive = {
          case _@ Ok(_: DeclareQueue, res: Option[_]) =>
            context become receiveBind(res.get.asInstanceOf[Queue.DeclareOk].getQueue)
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
        def receiveBind(queueName: String): Receive = {
          case _@ Ok(_: QueueBind, _: Option[_]) =>
            p trySuccess (consumer, queueName)
            handlerActor.foreach(consumer ! AddStatusListener(_))
          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
      }))

      init.foreach(request => consumer.tell(Record(request), proxy))
    })
    p.future
  }

  /**
   * Sets up an RPC server. The server starts processing incoming messages sent to an exchange and routing key and
   * replies to them according to a given behavior.
   *
   * The queue name can be `None` to create a One-To-Many (all servers/processors handle the request) or specified
   * for a One-To-Any (only one server/processor will handle the request).
   *
   * @param exchangeName the exchange name
   * @param routingKey the routing key
   * @param queueName the underlying queue name wrapped in a `Some` for One-To-Any behavior, or `None` to generate one
   *                  with a random suffix for One-To-Many behavior
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @param timeout an optional message timeout to avoid filling up the queue with old, already ignored, responses
   * @param process the function that processes `Delivery` messages and returns the reply for each one of them
   * @return a `Future` that is completed with the queue name when the server is started.
   */
  def createRpcServer(
    exchangeName: String,
    routingKey: String,
    queueName: Option[String] = None,
    handlerActor: Option[ActorRef] = None,
    timeout: Option[FiniteDuration] = None)(process: Delivery => Future[ProcessResult]): Future[String] = {

    val finalQueueName = queueName.getOrElse(generateName())

    val extraArgs: Map[String, AnyRef] = timeout match {
      case Some(t) => Map("x-message-ttl" -> Long.box(t.toMillis))
      case _ => Map.empty
    }

    val p = Promise[String]()

    declareExchange(exchangeName, handlerActor = handlerActor).foreach(_ => {

      val processor = new IProcessor {
        def process(delivery: Delivery) = process(delivery)
        def onFailure(delivery: Delivery, e: Throwable) = ProcessResult(None)
      }

      val exchangeParams = ExchangeParameters(exchangeName, passive = false, "topic", durable = true)
      val queueParams = QueueParameters(finalQueueName, passive = false, args = extraArgs)

      val rpcServerProps = RpcServer.props(processor, channelParams = Some(ChannelParameters(1)))

      rpcActor(rpcServerProps, rpcChannel => {
        handlerActor.foreach(rpcChannel ! AddStatusListener(_))
        val proxy = actorRefFactory.actorOf(Props(new Actor {
          def receive = {
            case _@ Ok(_: AddBinding, _) =>
              p trySuccess finalQueueName
            case error: Error =>
              handlerActor.foreach(_ forward error)
              p tryFailure error.reason
          }
        }))

        rpcChannel.tell(Record(AddBinding(Binding(exchangeParams, queueParams, routingKey))), proxy)
      })

    })

    p.future
  }

  /**
   * Creates an RPC client.
   *
   * @param handlerActor the actor to handle the channel state; it can receive the messages
   *                     [[com.github.sstone.amqp.ChannelOwner.Disconnected]],
   *                     [[com.github.sstone.amqp.ChannelOwner.Connected]] and [[com.github.sstone.amqp.Amqp.Error]]
   * @return a `Future` that is completed with the RPC producer actor when it is successfully bound.
   */
  def createRpcClient(handlerActor: Option[ActorRef] = None): Future[ActorRef] = {
    val p = Promise[ActorRef]()

    rpcActor(RpcClient.props(), rpcChannel => {

      val proxy = actorRefFactory.actorOf(Props(new Actor {
        def receive = {

          case com.github.sstone.amqp.ChannelOwner.Connected =>
            p trySuccess rpcChannel

          case error: Error =>
            handlerActor.foreach(_ forward error)
            p tryFailure error.reason
        }
      }))

      rpcChannel ! AddStatusListener(proxy)
    })

    p.future
  }
}
