package eu.shiftforward.adstax.ups.api

import scala.concurrent.Future

/**
 * Interface for clients capable of interacting with the User Profile Storage module, by retrieving updating or
 * deleting information about users.
 */
trait UserProfileStorageClient {

  /**
   * Retrieves the attributes of a user.
   *
   * @param userId the identifier of the user
   * @param clientId the identifier of the client
   * @return a `Future` with the user attributes wrapped in a `Some` if the user exists, `None` otherwise.
   */
  def get(userId: String, clientId: String): Future[Option[UserAttributes]]

  /**
   * Updates the attributes of a user.
   *
   * @param userId the identifier of the user
   * @param clientId the identifier of the client
   * @param attributes the attributes to update the user with
   * @param mergeStrategy the strategy to use when merging the attributes
   * @return a `Future ` with `true` if the user update was successful, `false` otherwise.
   */
  def update(
    userId: String,
    clientId: String,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy = JoinMergingStrategy): Future[Boolean]

  /**
   * Deletes the information about a user.
   *
   * @param userId the identifier of the user
   * @param clientId the identifier of the client
   * @return a `Future ` with `true` if the user deletion was successful, `false` otherwise.
   */
  def delete(userId: String, clientId: String): Future[Boolean]
}
