package eu.shiftforward.adstax.ups.api

import spray.json._

/**
 * A strategy to merge two sets of user attributes.
 */
sealed trait AttributeMergingStrategy

/**
 * Simple attribute merge strategy that updates the old attributes with the most recent ones. Attributes that were only
 * present in the old attributes are preserved.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (b, 1), (c, 1)]
 */
case object JoinMergingStrategy extends AttributeMergingStrategy

/**
 * Simple attribute merge strategy that overwrites the old attributes with new attributes, removing all old attributes.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (c, 1)]
 */
case object ReplaceMergingStrategy extends AttributeMergingStrategy

object AttributeMergingStrategy {

  implicit object AttributeMergingStrategyJsonFormat extends JsonFormat[AttributeMergingStrategy] {

    def read(json: JsValue) = json match {
      case JsString("join") => JoinMergingStrategy
      case JsString("replace") => ReplaceMergingStrategy
      case strategy => deserializationError(s"Invalid attribute merging strategy: ${strategy.compactPrint}")
    }

    def write(obj: AttributeMergingStrategy) = obj match {
      case JoinMergingStrategy => JsString("join")
      case ReplaceMergingStrategy => JsString("replace")
    }
  }
}
