package eu.shiftforward.adstax.scheduler.api.job

import scala.concurrent.duration._

import spray.json.DefaultJsonProtocol._
import spray.json._

import eu.shiftforward.adstax.scheduler.api.job.SchedulerJob.RepetitionConfig
import eu.shiftforward.adstax.scheduler.api.{ Continue, MissedExecutionBehavior }
import eu.shiftforward.adstax.util.InstanceLoader
import eu.shiftforward.apso.Logging
import eu.shiftforward.apso.config.Implicits._
import eu.shiftforward.apso.config.LazyConfigFactory
import eu.shiftforward.apso.json.JsonFormatBuilder

/**
 * A job able to be scheduled.
 */
trait SchedulerJob {

  /**
   * The identifier of the job
   */
  def id: String

  /**
   * The description of the job
   */
  def action: String

  /**
   * The initial delay before the job is first executed
   */
  def initialDelay: FiniteDuration

  /**
   * The repetition configuration for the job. When it is `None`, it is assumed
   * that the job doesn't repeat. Besides the repetition interval, it is
   * possible to define what should happen when some executions were missed.
   */
  def repeat: Option[RepetitionConfig]
}

object SchedulerJob extends Logging {

  /**
   * Defines the repetition configuration for the task.
   *
   * @param interval the duration in which to repeat the scheduled action
   * @param missedExecutionBehavior What to do when loading repeating tasks whose last run indicates that it missed
   *                                some executions. That is, `currentTime() - lastRun > repeat`. This is important
   *                                on situations of failure or restart, where the scheduler service has to skip
   *                                executions. Upon loading the persisted tasks, the service can take into account
   *                                this setting, and act accordingly.
   */
  final case class RepetitionConfig(
    interval: FiniteDuration,
    missedExecutionBehavior: MissedExecutionBehavior = Continue)

  object RepetitionConfig {
    implicit object MillisDurationJsonFormat extends JsonFormat[FiniteDuration] {
      def read(json: JsValue) = json.convertTo[Long].millis
      def write(duration: FiniteDuration) = duration.toMillis.toJson
    }

    implicit val jsonFormat: RootJsonFormat[RepetitionConfig] = jsonFormat2(RepetitionConfig.apply)
  }

  object JsonProtocol {
    import RepetitionConfig._

    val baseFormatBuilder = JsonFormatBuilder()
      .field[String]("id")
      .field[String]("action")
      .field[FiniteDuration]("initialDelay")
      .optionalField[RepetitionConfig]("repeat")
  }

  implicit object SchedulerJobJsonFormat extends RootJsonFormat[SchedulerJob] {

    private[this] val registry: Map[String, JsonFormat[SchedulerJob]] =
      LazyConfigFactory.load.getConfig("adstax.scheduler")
        .getMap[String]("job-types")
        .flatMap {
          case (action, jfClass) =>
            InstanceLoader.getInstance[SchedulerJobCompanion[SchedulerJob]](jfClass) match {
              case Some(companion) => Some(action -> companion.jobJsonFormat)
              case None =>
                log.info(s"Could not load companion object $jfClass for action $action")
                None
            }
        }

    def read(v: JsValue): SchedulerJob = {
      v.asJsObject.fields.get("action") match {
        case Some(JsString(action)) if registry.contains(action) =>
          v.convertTo[SchedulerJob](registry(action))
        case Some(JsString(action)) =>
          deserializationError(s"Unknown or unsupported: 'action' field value: $action")
        case _ =>
          deserializationError("JSON object not recognized: missing 'action' field for a valid SchedulerAction!")
      }
    }

    def write(a: SchedulerJob): JsValue = a match {
      case a: SchedulerJob if registry.contains(a.action) =>
        a.toJson(registry(a.action))
      case other =>
        serializationError(s"Cannot serialize unrecognized Scheduler Action: ${other.getClass}!" +
          s" Does the JSON Format Registry contain a '${a.action}' entry for that Scheduler Action!?")
    }
  }
}

/**
 * A trait for companion objects of `SchedulerJob` implementations. Provides JSON formats used to serialize and
 * deserialize the different types of jobs.
 *
 * @tparam T the type of the job
 */
trait SchedulerJobCompanion[T <: SchedulerJob] {
  def jobJsonFormat: RootJsonFormat[T]
}
