package eu.shiftforward.adstax.recommender.api

import fommil.sjs.FamilyFormats._
import spray.json.RootJsonFormat

/**
 * A specification of a recommendation query.
 */
sealed trait RecommendationQuery {
  def uidToSegment: Option[String]
  def clientId: Option[String]
  def siteId: Option[String]
  def getId: String
  def getNonPersonalizedId = Recommendation.getId(None, None)
  def features: Option[Features]
}

/**
 * A query to obtain a recommendation for an user.
 *
 * @param userId the identifier of the user
 * @param clientId the identifier of the client
 * @param siteId the identifier of the site
 * @param features the features related to this query
 */
case class UserRecommendationQuery(
    userId: String,
    override val clientId: Option[String],
    override val siteId: Option[String],
    override val features: Option[Features] = None) extends RecommendationQuery {

  def uidToSegment = Some(userId)
  def getId = (Some(userId), None).toString()
}

object UserRecommendationQuery {
  implicit val userRecommendationQueryFormat = jsonFormat4(UserRecommendationQuery.apply)
}

/**
 * A query to obtain a recommendation for a product.
 *
 * @param productId the identifier of the product
 * @param clientId the identifier of the client
 * @param siteId the identifier of the site
 * @param uidToSegment the user identififer to be used for segmentation
 * @param features the features related to this query
 */
case class ProductRecommendationQuery(
    productId: String,
    override val clientId: Option[String],
    override val siteId: Option[String],
    override val uidToSegment: Option[String] = None,
    override val features: Option[Features] = None) extends RecommendationQuery {

  def getId = (None, Some(productId)).toString()
}

object ProductRecommendationQuery {
  implicit val productRecommendationQueryFormat = jsonFormat5(ProductRecommendationQuery.apply)
}

object RecommendationQuery {
  implicit val recommendationQueryJsonFormat: RootJsonFormat[RecommendationQuery] = familyFormat
}
