package eu.shiftforward.adstax.productfeeder.api

import java.net.URI

import spray.json.DefaultJsonProtocol._

import eu.shiftforward.apso.json.ExtraJsonProtocol._

/**
 * A request to be sent to the Product Feeder module.
 */
sealed trait ProductFeederRequest

/**
 * A request to retrieve information about a product.
 *
 * @param query the query definition for the product
 */
case class GetProduct(query: ProductQuery) extends ProductFeederRequest

object GetProduct {
  implicit val getProductJsonFormat = jsonFormat1(GetProduct.apply)
}

/**
 * A request to update the information about a product.
 *
 * @param clientId the identifier of the client
 * @param siteId the optional identifier of the site. If set to `null` the product will be updated at a client level; if
 *              set it will update at both levels.
 * @param product the updated product information
 */
case class UpdateProduct(clientId: String, siteId: String, product: ProductItem) extends ProductFeederRequest

object UpdateProduct {
  implicit val updateProductJsonFormat = jsonFormat3(UpdateProduct.apply)
}

/**
 * A request to delete the information about a product.
 *
 * @param clientId the identifier of the client
 * @param siteId the optional identifier of the site. If set to `null` the product will be updated at a client level;
 *               otherwise it will be updated at the site level.
 * @param productId the identifier of the product
 */
case class DeleteProduct(clientId: String, siteId: String, productId: String) extends ProductFeederRequest

object DeleteProduct {
  implicit val deleteProductsFormat = jsonFormat3(DeleteProduct.apply)
}

/**
 * A request to download and import information from a product feed file.
 *
 * @param clientId the identifier of the client
 * @param siteId the optional identifier of the site. If set to `null` the product will be deleted at a client level;
 *               otherwise it will be deleted at the site level.
 * @param uri a URL pointing to the product feed
 * @param mimeType the MIME type of the product feed file
 */
case class DownloadFeed(clientId: String, siteId: String, uri: URI, mimeType: MimeType)
  extends ProductFeederRequest

object DownloadFeed {
  implicit val downloadFeedFormat = jsonFormat4(DownloadFeed.apply)
}
