package eu.shiftforward.adstax.storage.rpc

import scala.concurrent.{ ExecutionContext, Future }

import akka.actor.ActorRefFactory

import eu.shiftforward.adstax.config
import eu.shiftforward.adstax.config.ConfigConvertProtocol._
import eu.shiftforward.adstax.storage.UserAttributes
import eu.shiftforward.adstax.storage.UserProfileStorageClient
import eu.shiftforward.adstax.storage.rpc.UserProfileStorageRequest._
import eu.shiftforward.adstax.storage.rpc.UserProfileStorageResponse._
import eu.shiftforward.adstax.storage.mergers.{ AttributeMergingStrategy, JoinMergingStrategy }
import eu.shiftforward.adstax.util.{ RmqRpcJsonClient, RmqRpcJsonClientTypeDescriptor }
import eu.shiftforward.apso.config.LazyConfigFactory
import pureconfig.syntax._

/**
 * The RPC client to interact with the user profile storage. It implements a RabbitMQ RPC protocol for issuing get, put,
 * delete and update requests and exposes a type-safe interface for those operations.
 */
class UserProfileStorageRmqRpcClient(
  actorRefFactory: ActorRefFactory,
  val rmqConfig: config.RabbitMQ = LazyConfigFactory.load.getConfig("adstax.user-profile-storage.rabbitmq").to[config.RabbitMQ].get,
  val rpcClientConfig: config.RpcClient = LazyConfigFactory.load.getConfig("adstax.user-profile-storage.rpc").to[config.RpcClient].get)(implicit val ec: ExecutionContext)
    extends RmqRpcJsonClient with UserProfileStorageClient {

  def rmqActorRefFactory = actorRefFactory

  implicit val getTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[UserProfileGet, GetResponse] {}
  implicit val deleteTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[UserProfileDelete, DeleteResponse] {}
  implicit val updateTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[UserProfileUpdate, UpdateResponse] {}

  implicit val getRoutingKey = UserProfileStorageRequest.GetTypeRoutingKey
  implicit val deleteRoutingKey = UserProfileStorageRequest.DeleteTypeRoutingKey
  implicit val updateRoutingKey = UserProfileStorageRequest.UserProfileUpdateTypeRoutingKey

  import UserProfileStorageRequest.JsonProtocol.{ deleteFormat, getFormat, updateFormat }
  import UserProfileStorageResponse.JsonProtocol.{ deleteResponseFormat, getResponseFormat, updateResponseFormat }

  def get(userId: String, clientId: String): Future[Option[UserAttributes]] =
    for {
      req <- dispatchRequest[UserProfileGet, GetResponse](UserProfileGet(userId, clientId))
    } yield (req.attributes)

  def update(
    userId: String,
    clientId: String,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy = JoinMergingStrategy): Future[Boolean] =
    for {
      req <- dispatchRequest[UserProfileUpdate, UpdateResponse](UserProfileUpdate(userId, clientId, attributes, mergeStrategy))
    } yield (req.success)

  def delete(userId: String, clientId: String): Future[Boolean] =
    for {
      req <- dispatchRequest[UserProfileDelete, DeleteResponse](UserProfileDelete(userId, clientId))
    } yield (req.success)
}
