package eu.shiftforward.adstax.storage.mergers

import spray.json.{ JsString, JsValue, RootJsonFormat }

/**
 * Abstract strategy to merge two sets of attributes,
 */
sealed trait AttributeMergingStrategy

/**
 * Simple attribute merge strategy that updates the old attributes with the most recent ones.
 * Attributes that were only present in the old attributes are preserved.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (b, 1), (c, 1)]
 */
case object JoinMergingStrategy extends AttributeMergingStrategy

/**
 * Simple attribute merge strategy that overwrites the old attributes with new attributes, removing all old
 * attributes.
 *
 * Example:
 * [(a, 1), (b, 1)] * [(a, 3), (c, 1)] = [(a, 3), (c, 1)]
 */
case object ReplaceMergingStrategy extends AttributeMergingStrategy

object AttributeMergingStrategy {
  object JsonProtocol {
    implicit object AttributeMergingStrategyJsonFormat extends RootJsonFormat[AttributeMergingStrategy] {
      override def read(json: JsValue): AttributeMergingStrategy = json match {
        case JsString("join") => JoinMergingStrategy
        case JsString("replace") => ReplaceMergingStrategy
        case x => throw new IllegalArgumentException("Invalid attribute merging strategy: " + x)
      }

      override def write(obj: AttributeMergingStrategy): JsValue = obj match {
        case JoinMergingStrategy => JsString("join")
        case ReplaceMergingStrategy => JsString("replace")
      }
    }
  }
}
