package eu.shiftforward.adstax.scheduler.action

import scala.concurrent.duration.FiniteDuration

import spray.json._
import spray.json.DefaultJsonProtocol._

import eu.shiftforward.adstax.scheduler.action.SchedulerAction.RepetitionConfig
import eu.shiftforward.apso.config.Implicits._
import eu.shiftforward.apso.config.LazyConfigFactory
import eu.shiftforward.apso.json.ExtraTimeJsonProtocol

/**
 * Represents an action capable to be scheduled.
 */
trait SchedulerAction {

  /**
   * Returns the id of the action.
   */
  def id: String

  /**
   * Returns the description of the action.
   */
  def action: String

  /**
   * Returns the duration in which to run the action.
   */
  def in: FiniteDuration

  /**
   * Returns the duration in which to repeat the scheduled action, or None if it's an one-shot task.
   */
  def repeat: Option[RepetitionConfig]
}

/**
 * Conversions to and from JSON for [[SchedulerAction]].
 */
object SchedulerAction {

  /**
   * Defines the repetition configuration for the task
   *
   * @param interval the duration in which to repeat the scheduled action
   * @param missedExecutionBehavior What to do when loading repeating tasks whose last run indicates that it missed
   *                                some executions. That is, `currentTime() - lastRun > repeat`. This is important
   *                                on situations of failure or restart, where the scheduler service has to skip
   *                                executions. Upon loading the persisted tasks, the service can take into account
   *                                this setting, and act accordingly.
   */
  final case class RepetitionConfig(interval: FiniteDuration, missedExecutionBehavior: MissedExecutionBehavior = Continue)
  object RepetitionConfig extends ExtraTimeJsonProtocol {
    implicit val jsonFormat: JsonFormat[RepetitionConfig] = jsonFormat2(RepetitionConfig.apply)
  }

  sealed trait MissedExecutionBehavior {
    def name: String
  }
  object MissedExecutionBehavior {
    implicit val jsonFormat: JsonFormat[MissedExecutionBehavior] = new JsonFormat[MissedExecutionBehavior] {
      override def write(obj: MissedExecutionBehavior): JsValue = JsString(obj.name)
      override def read(json: JsValue): MissedExecutionBehavior = json match {
        case JsString("continue") => Continue
        case JsString("executeAll") => ExecuteAll
        case JsString(other) =>
          deserializationError(s"Impossible to read MissedExecutionBehavior: unexpected value '$other'!")
        case _ =>
          deserializationError("Impossible to read MissedExecutionBehavior: unexpected JSON type!")
      }
    }
  }
  case object Continue extends MissedExecutionBehavior {
    val name = "continue"
  }
  case object ExecuteAll extends MissedExecutionBehavior {
    val name = "executeAll"
  }

  object JsonProtocol {
    val jsonFormatRegistry: Map[String, JsonFormat[SchedulerAction]] =
      LazyConfigFactory.load.getConfig("adstax.sdk.scheduler").getMap[String]("json-formats").mapValues { jfClass =>
        getClass.getClassLoader.loadClass(jfClass).newInstance().asInstanceOf[JsonFormat[SchedulerAction]]
      }

    /**
     * The JsonProtocol object, with conversions to and from JSON.
     */
    implicit object SchedulerActionJsonProtocol extends RootJsonFormat[SchedulerAction] {
      def read(v: JsValue): SchedulerAction = {
        v.asJsObject.fields.get("action") match {
          case Some(JsString(action)) if jsonFormatRegistry.contains(action) =>
            v.convertTo[SchedulerAction](jsonFormatRegistry(action))
          case Some(JsString(action)) => deserializationError(s"Unknown or unsupported: 'action' field value: $action")
          case _ => deserializationError("JSON object not recognized: missing 'action' field for a valid SchedulerAction!")
        }
      }

      def write(a: SchedulerAction): JsValue = a match {
        case a: SchedulerAction if jsonFormatRegistry.contains(a.action) => a.toJson(jsonFormatRegistry(a.action))
        case other => serializationError(s"Cannot serialize unrecognized Scheduler Action: ${other.getClass}!" +
          s" Does the JSON Format Registry contain a '${a.action}' entry for that Scheduler Action!?")
      }
    }
  }
}
