package eu.shiftforward.adstax.tracking

import akka.actor._
import com.github.sstone.amqp.Amqp.Delivery
import com.typesafe.config.Config
import eu.shiftforward.adstax.util.RabbitMQUtilAsync
import eu.shiftforward.apso.config.LazyConfigFactory
import spray.json._

import scala.concurrent.ExecutionContext

/**
 * Base trait containing functionality for consuming tracking events from an AdStax Event Tracker instance. The
 * processing step is defined by a `Props` which creates worker actors able to process JSON events. The trait also
 * allows configuring the number of workers to instantiate, providing support for parallel consuming of events.
 */
trait TrackingListener extends RabbitMQUtilAsync {

  /**
   * The `ActorRef` factory, such as an `ActorSystem` or `ActorContext`, used to create the actors needed for tracking.
   */
  def actorRefFactory: ActorRefFactory
  def rmqActorRefFactory = actorRefFactory

  implicit def ec: ExecutionContext

  /**
   * The tracking configuration.
   */
  def trackingConfig: Config

  /**
   * A `Props` encapsulating the creation of worker actors. The actor should be able to receive and process JSON events
   * in the form of [[spray.json.JsValue]] objects.
   */
  def processorProps: Props

  /**
   * The number of worker actors to instantiate. The instantiated workers will process events in parallel.
   */
  def numProcessors: Int

  /**
   * The name of the AMQP exchange from which the listener will consume events.
   */
  lazy val exchangeName = trackingConfig.getString("exchange-name")

  /**
   * The routing key from which the listener will consume events.
   */
  lazy val routingKey = trackingConfig.getString("routing-key")

  private[this] class JsonDeserializerActor(queueName: String) extends Actor {
    val processor = context.actorOf(processorProps)
    setListeningQueue(self, exchangeName, routingKey, Some(queueName))

    def receive = {
      case Delivery(_, _, _, body) => processor ! new String(body, "UTF-8").parseJson
    }
  }

  declareQueue(exchangeName, routingKey).foreach { queueName =>
    for (_ <- 1 to numProcessors)
      actorRefFactory.actorOf(Props(new JsonDeserializerActor(queueName)))
  }
}

/**
 * An actor able to consume tracking events from an AdStax Event Tracker instance.
 * @param processorProps a `Props` encapsulating the creation of worker actors
 * @param numProcessors the number of worker actors to instantiate
 * @param rmqConfig the RabbitMQ configuration
 * @param trackingConfig the tracking configuration
 */
class TrackingListenerActor(
    val processorProps: Props,
    val numProcessors: Int,
    val rmqConfig: Config,
    val trackingConfig: Config) extends Actor with TrackingListener {

  def actorRefFactory = context
  def ec = context.dispatcher

  def receive = Actor.emptyBehavior
}

/**
 * Companion object containing factory methods for `TrackingListenerActor` actors.
 */
object TrackingListenerActor {

  /**
   * Creates a `Props` for creating `TrackingListenerActor` actors with the default settings.
   * @param processorProps a `Props` encapsulating the creation of worker actors
   * @param numProcessors the number of worker actors to instantiate
   * @param rmqConfig the RabbitMQ configuration
   * @param trackingConfig the tracking configuration
   * @return a `Props` for creating `TrackingListenerActor` actors.
   */
  def props(
    processorProps: Props,
    numProcessors: Int = 1,
    rmqConfig: Config = LazyConfigFactory.load.getConfig("adstax.event-tracker.rabbitmq"),
    trackingConfig: Config = LazyConfigFactory.load.getConfig("adstax.event-tracker.http-tracking")) =
    Props(new TrackingListenerActor(processorProps, numProcessors, rmqConfig, trackingConfig))
}
