package eu.shiftforward.adstax.storage

import scala.concurrent.Future

import eu.shiftforward.adstax.storage.mergers.{ AttributeMergingStrategy, JoinMergingStrategy }

/**
 * Base trait for user profile storage clients. Exposes all the necessary methods to deal with a user profile storage.
 */
trait UserProfileStorageClient {
  /**
   * Retrives the attributes of an user.
   *
   * @param userId the ID of the user
   * @param clientId the ID of the client
   * @return a `Future` with the user attributes wrapped in a `Some` if it exists, `None` otherwise.
   */
  def get(userId: String, clientId: String): Future[Option[UserAttributes]]

  /**
   * Updates the user details.
   *
   * @param userId the ID of the user
   * @param clientId the ID of the client
   * @param attributes the attributes to update the user with
   * @param mergeStrategy the strategy to use when merging the attributes
   * @return a `Future ` with `true` if the user update was successful, `false` otherwise.
   */
  def update(
    userId: String,
    clientId: String,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy = JoinMergingStrategy): Future[Boolean]

  /**
   * Deletes the user details.
   *
   * @param userId the ID of the user
   * @param clientId the ID of the client
   * @return a `Future ` with `true` if the user delete was successful, `false` otherwise.
   */
  def delete(userId: String, clientId: String): Future[Boolean]
}
