package eu.shiftforward.adstax.scheduler

import spray.json._
import spray.json.lenses.JsonLenses._

import eu.shiftforward.adstax.scheduler.action.SchedulerAction
import eu.shiftforward.adstax.scheduler.action.SchedulerAction.JsonProtocol._

/**
 * Represents the supported responses from a given [[eu.shiftforward.adstax.scheduler.rpc.SchedulingRequestMessage]].
 */
trait SchedulingResponseMessage
sealed trait JobStatus extends SchedulingResponseMessage {
  def status: String
}
case class JobsStatus(jobs: Map[String, JobStatus]) extends SchedulingResponseMessage
case class Scheduled(action: SchedulerAction) extends JobStatus {
  val status = "scheduled"
}
case class Cancelled(actionId: String) extends JobStatus {
  val status = "cancelled"
}

object SchedulerOperationResult {
  /**
   * The set of JSON formats required to serialized and deserialize the reply messages
   */
  object JsonProtocol extends DefaultJsonProtocol {

    // reads and writes Scheduled
    implicit val scheduledFormat: RootJsonFormat[Scheduled] = new RootJsonFormat[Scheduled] {
      private val f = jsonFormat(Scheduled, "action")
      override def read(json: JsValue): Scheduled = f.read(json)
      override def write(obj: Scheduled): JsValue = f.write(obj).update('status ! set("scheduled"))
    }

    // reads and writes Cancelled
    implicit val cancelledFormat = new RootJsonFormat[Cancelled] {
      private val f = jsonFormat(Cancelled, "actionId")
      override def read(json: JsValue): Cancelled = f.read(json)
      override def write(obj: Cancelled): JsValue = f.write(obj).update('status ! set("cancelled"))
    }

    // reads and writes JobsStatus
    implicit val jobsStatusFormat = new RootJsonFormat[JobsStatus] {
      override def write(js: JobsStatus) = JsObject(
        "type" -> JsString("jobs-status"),
        "jobs" -> js.jobs.toJson)

      // case class JobsStatus(jobs: List[SchedulerStatus]) extends JobsStatusReply
      override def read(json: JsValue): JobsStatus = json.asJsObject.fields("jobs") match {
        case JsObject(jobs) => JobsStatus(jobs.mapValues(_.convertTo[JobStatus]))
        case somethingElse => deserializationError(
          s"JobsStatus mandatory 'jobs' parameter is missing or has an invalid type. Got: $somethingElse.")
      }
    }

    // reads and writes JobStatus
    implicit val jobStatusFormat: RootJsonFormat[JobStatus] = new RootJsonFormat[JobStatus] {
      def write(js: JobStatus) = js match {
        case scheduled: Scheduled => scheduledFormat.write(scheduled)
        case cancelled: Cancelled => cancelledFormat.write(cancelled)
      }
      def read(json: JsValue): JobStatus = json.asJsObject.getFields("status") match {
        case Seq(JsString(status)) => status match {
          case "scheduled" => scheduledFormat.read(json)
          case "cancelled" => cancelledFormat.read(json)
        }
      }
    }
  }
}
