package eu.shiftforward.adstax.scheduler

import javax.script._
import spray.json._

/**
 * Functions in this object will be accessible in the JavaScript engine used for `RecurrentScript`.
 */
object JavaScriptUtils {
  import eu.shiftforward.apso.http.W

  /**
   * Checks if the resource pointed by the given `url` exists by performing an `HEAD` HTTP request.
   * Exceptions thrown by the HTTP client will be silently discarded and the method will return `false`.
   *
   * @param url the url of the resource to check
   * @return true if the resource exists, false otherwise.
   */
  def httpResourceExists(url: String): Boolean = {
    try {
      val statusCode = W.head(url).getStatusCode
      statusCode >= 200 && statusCode < 400
    } catch {
      case _: Throwable => false
    }
  }
}

/**
 * A script to be run recurrently, reusing the return values as its inner state. In order to reuse the return value in
 * the next iteration, a JSON object with the `nextInput` field must be returned. In that case, the value that is
 * effectively used as the return value should be present in the `returnValue` field. The value that is returned in the
 * `nextInput` field is available on the next execution of the script in the `input` variable.
 *
 * @param script a JavaScript script that implements the body of a function that accepts a JSON value as argument in the
 *               `input` parameter
 * @param inputValue the initial JSON value that is supplied to the first execution of the script. Its default
 *                   value is an empty JSON object. This field will be mutated after the script is run, with the new
 *                   input value returned by executing the script.
 */
class RecurrentScript(
    private[scheduler] val script: String,
    private[scheduler] var inputValue: JsValue = JsObject()) {
  import RecurrentScript._

  private[this] val f = {
    val manager = new ScriptEngineManager()
    Option(manager.getEngineByName("nashorn")) match {

      case Some(engine: Compilable with Invocable) =>
        engine.compile(prelude.format(script)).eval()
        engine.getInterface(classOf[RecurrentScriptContext])

      case _ =>
        throw new UnsupportedOperationException("Job definition using JavaScript is not supported by this system")
    }
  }

  /**
   * Executes the recurrent script with the next input value, returning a JsValue.
   * @return the value that is returned from the current execution of the script.
   */
  def execute(): JsValue = {
    val result = f.__wrapExecuteScript(inputValue.compactPrint).parseJson
    result match {
      case JsObject(fields) if fields.contains("nextInput") && fields.contains("returnValue") =>
        inputValue = fields("nextInput")
        fields("returnValue")
      case _ => result
    }
  }
}

/**
 * Contexts for running the JavaScript scripts.
 */
object RecurrentScript {
  private[RecurrentScript] val prelude =
    """
      |var javaUtils = Java.type('eu.shiftforward.adstax.scheduler.JavaScriptUtils');
      |var utils = {};
      |utils.httpResourceExists = javaUtils.httpResourceExists;
      |utils.isEmpty = function(x) {
      |  return (typeof x === 'object' && Object.keys(x).length === 0) || typeof x === 'undefined';
      |};
      |function __wrapExecuteScript(input) { return JSON.stringify(executeScript(JSON.parse(input))); }
      |function executeScript(input) { %s }
    """.stripMargin

  /**
   * Defines the interface for the JavaScript scripts.
   */
  trait RecurrentScriptContext {
    /**
     * Converts the return value of the `executeScript` function to a string that is parseable to JSON in Scala.
     *
     * @param input the input value to the `executeScript` function
     * @return a string with the returned value.
     */
    def __wrapExecuteScript(input: String): String
  }

  object JsonProtocol {
    implicit object RecurrentScriptJsonFormat extends RootJsonFormat[RecurrentScript] {
      def write(rs: RecurrentScript) = JsObject(
        "script" -> JsString(rs.script),
        "input" -> rs.inputValue)

      def read(value: JsValue) = {
        val jsObject = value.asJsObject
        jsObject.getFields("script") match {
          case Seq(JsString(script)) =>
            new RecurrentScript(script, jsObject.fields.getOrElse("input", JsObject()))
          case _ =>
            throw new DeserializationException("The mandatory parameter (script) of a message definition is missing or has" +
              "an invalid type")
        }
      }
    }
  }
}
