package eu.shiftforward.adstax.recommender

import spray.json._

/**
 * Represents a recommendation.
 *
 * @param userId user ID
 * @param productId product ID (query)
 * @param clientId client ID
 * @param siteId site ID
 * @param recommendedProductIds recommended product IDs
 * @param timestamp timestamp
 */
case class Recommendation(
    userId: Option[String],
    productId: Option[String],
    clientId: Option[String] = None,
    siteId: Option[String] = None,
    recommendedProductIds: Seq[String] = List.empty,
    timestamp: Option[String] = None) {

  /**
   * The recommendation ID
   */
  val id = (userId, productId).toString

}

object Recommendation {
  def getId(r: Recommendation): String = Recommendation.getId(r.userId, r.productId)

  def getId(userId: Option[String], productId: Option[String]): String = (userId, productId).toString

  object JsonProtocol extends DefaultJsonProtocol with NullOptions {
    implicit object RecommendationJsonFormat extends RootJsonFormat[Recommendation] {
      override def read(json: JsValue): Recommendation = {
        val fields = json.asJsObject.fields
        new Recommendation(
          fields.get("uid").map {
          case JsString(value) => Some(value)
          case _ => None
        }.get,
          fields.get("pid").map {
          case JsString(value) => Some(value)
          case _ => None
        }.get,
          fields.get("clientId").map {
          case JsString(value) => Some(value)
          case _ => None
        }.get,
          fields.get("siteId").map {
          case JsString(value) => Some(value)
          case _ => None
        }.get,
          fields.get("recommendedProducts").map {
          case JsArray(values: Vector[JsString @unchecked]) => values.map(_.value)
          case _ => List.empty
        }.get,
          fields.get("timestamp").map {
          case JsString(value) => Some(value)
          case _ => None
        }.get)
      }

      override def write(rec: Recommendation): JsValue = JsObject(
        "timestamp" -> rec.timestamp.toJson,
        "clientId" -> rec.clientId.toJson,
        "siteId" -> rec.siteId.toJson,
        "recommendedProducts" -> rec.recommendedProductIds.toJson,
        "uid" -> rec.userId.toJson,
        "pid" -> rec.productId.toJson)
    }
  }
}
