package eu.shiftforward.adstax.util

import akka.pattern.ask
import akka.util.Timeout
import com.github.sstone.amqp.Amqp.Publish
import com.github.sstone.amqp.RpcClient.Request
import com.github.sstone.amqp.{ Amqp, RpcClient }
import com.typesafe.config.Config
import eu.shiftforward.adstax.util.Rpc.JsonProtocol._
import spray.json._
import spray.json.DefaultJsonProtocol._

import scala.concurrent.duration._
import scala.concurrent.{ ExecutionContext, Future }

/**
 * Describes the evidence of the expected response type from a given request
 *
 * @tparam Req the type of the request message with a JsonWriter
 * @tparam Resp the response type that the callback returns
 */
trait RmqRpcJsonClientTypeDescriptor[Req, Resp]

/**
 * The Json Routing Key for a specific type.
 *
 * @tparam Req the type which the routing key refers to
 */
trait RmqRpcJsonTypeRoutingKey[Req] {
  def value: String
}

/**
 * This trait abstracts the creation and communication logic of a Rpc client.
 *
 */
trait RmqRpcJsonClient extends RabbitMQUtilAsync {

  implicit def ec: ExecutionContext

  def rpcClientConfig: Config

  implicit val timeout = Timeout(10.seconds)

  private[this] val clientExchangeName = rpcClientConfig.getString("exchange-name")
  private[this] val clientRoutingKeyPrefix = rpcClientConfig.getString("routing-key-prefix")

  private val clientActor = declareExchange(clientExchangeName).flatMap(_ => createRpcClient())

  /**
   * Dispatches a request message from the client and returns a Future of a response
   *
   * @param req the request message
   * @param f the callback handler for the response
   * @return a Future of the return of the callback of the rpc response
   */
  def dispatch[Req, Resp](req: Req, routingKey: String)(f: Amqp.Delivery => Either[Rpc.Error, Resp])(implicit requestJsonWriter: JsonWriter[Req]): Future[Resp] = {

    clientActor.flatMap { client =>
      (client ? Request(Publish(clientExchangeName, clientRoutingKeyPrefix + routingKey, req.toJson.compactPrint.getBytes("UTF-8")))).flatMap {
        case response: RpcClient.Response =>
          f(response.deliveries.head) match {
            case Left(Rpc.Error(msg)) =>
              Future.failed(new Exception("[RmqRpcJsonClient] Error: " + msg))
            case Right(response) =>
              Future.successful(response)
          }
        case error: RpcClient.Undelivered =>
          Future.failed(new Exception("[RmqRpcJsonClient] Undelivered Message:" + error.msg))
      }
    }
  }

  /**
   * Dispatches a request message with a Future of a concrete response type.
   *
   * @param request the request message
   * @param td the evidence to support the serialization of a concrete type Resp
   * @param requestJsonWriter the json writer for type ```Req```
   * @param responseJsonReader the json reader for type ```Resp```
   * @tparam Req the type of the request message with a JsonWriter
   * @tparam Resp the response type that the callback returns
   * @return a Future of the expected response message
   */
  def dispatchRequest[Req, Resp](request: Req)(implicit
    td: RmqRpcJsonClientTypeDescriptor[Req, Resp],
    trk: RmqRpcJsonTypeRoutingKey[Req],
    requestJsonWriter: JsonWriter[Req],
    responseJsonReader: JsonFormat[Resp]): Future[Resp] =

    dispatch(request, trk.value) { response =>
      new String(response.body, "UTF-8").parseJson.convertTo[Either[Rpc.Error, Resp]]
    }
}
