package eu.shiftforward.adstax.storage.rpc

import eu.shiftforward.adstax.storage.UserAttributes
import eu.shiftforward.adstax.storage.UserAttributes.JsonProtocol._
import eu.shiftforward.adstax.storage.mergers.AttributeMergingStrategy
import eu.shiftforward.adstax.storage.mergers.AttributeMergingStrategy.JsonProtocol._
import eu.shiftforward.adstax.util.RmqRpcJsonTypeRoutingKey
import spray.json.DefaultJsonProtocol._

/**
 * Base trait for user-profile-storage request messages.
 */
sealed trait UserProfileStorageRequest

/**
 * Companion object containing the definition of the various `KeyValueStoreRequest` types, along with their JSON format
 * and associated routing keys.
 */
object UserProfileStorageRequest {

  /**
   * Message sent to the user-profile-storage requesting the value associated with a key.
   * @param userId the user ID to lookup
   * @param clientId the client ID of the user to lookup
   */
  case class UserProfileGet(
    userId: String,
    clientId: String) extends UserProfileStorageRequest

  /**
   * Message sent to the user-profile-storage requesting the updating or creation of a new key-value entry.
   * @param userId the user ID to update
   * @param attributes the attributes to insert on the storage
   * @param clientId the client ID of the user to update
   * @param mergeStrategy the strategy to use when merging the attributes
   */
  case class UserProfileUpdate(
    userId: String,
    clientId: String,
    attributes: UserAttributes,
    mergeStrategy: AttributeMergingStrategy) extends UserProfileStorageRequest

  /**
   * Message sent to the user-profile-storage requesting the deletion of a key-value entry.
   * @param userId the user ID to delete
   * @param clientId the client ID of the user to delete
   */
  case class UserProfileDelete(
    userId: String,
    clientId: String) extends UserProfileStorageRequest

  /**
   * Object defining the routing key to use for `Get` requests.
   */
  object GetTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[UserProfileGet] {
    def value = "get.json"
  }

  /**
   * Object defining the routing key to use for `Update` requests.
   */
  object UserProfileUpdateTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[UserProfileUpdate] {
    def value = "update.json"
  }

  /**
   * Object defining the routing key to use for `Delete` requests.
   */
  object DeleteTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[UserProfileDelete] {
    def value = "delete.json"
  }

  object JsonProtocol {
    /**
     * The JSON format for `Get` messages.
     */
    implicit val getFormat = jsonFormat2(UserProfileGet)

    /**
     * The JSON format for `Update` messages.
     */
    implicit val updateFormat = jsonFormat4(UserProfileUpdate)

    /**
     * The JSON format for `Delete` messages.
     */
    implicit val deleteFormat = jsonFormat2(UserProfileDelete)
  }
}
