package eu.shiftforward.adstax.storage

import spray.json.DefaultJsonProtocol._
import spray.json.{ JsArray, JsNumber, RootJsonFormat, _ }

import eu.shiftforward.adstax.storage.UserAttributes._
import eu.shiftforward.apso.Implicits._

case class UserAttributes(attributes: Map[String, AttributeValue]) {
  def toSymbolTable: Map[String, Any] = attributes.mapValues(_.toValue)
  def merge(that: UserAttributes): UserAttributes = {
    UserAttributes(this.attributes.twoWayMerge(that.attributes)(_ merge _))
  }
}

object UserAttributes {
  sealed trait AttributeValue {
    def toValue: Any
    def toJson: JsValue
    def merge(that: AttributeValue): AttributeValue = (this, that) match {
      case (MapAttrValue(thisMap), MapAttrValue(thatMap)) =>
        MapAttrValue(thisMap.twoWayMerge(thatMap)((js1, js2) => js1.merge(js2)))
      case (ArrayAttrValue(thisSeq), ArrayAttrValue(thatSeq)) => ArrayAttrValue(thisSeq ++ thatSeq)
      case _ => that
    }
  }

  case class StringAttrValue(str: String) extends AttributeValue {
    def toValue: String = str
    def toJson = JsString(str)
  }

  case class NumericAttrValue(num: Double) extends AttributeValue {
    def toValue: Double = num
    def toJson = JsNumber(num)
  }

  case class BooleanAttrValue(bool: Boolean) extends AttributeValue {
    def toValue: Boolean = bool
    def toJson = JsBoolean(bool)
  }

  case class GeoAttrValue(lat: Double, long: Double) extends AttributeValue {
    def this(latLong: (Double, Double)) = this(latLong._1, latLong._2)
    def toValue: (Double, Double) = (lat, long)
    def toJson = JsObject(Map("lat" -> JsNumber(lat), "long" -> JsNumber(long)))
  }

  case class ArrayAttrValue(arr: Seq[AttributeValue]) extends AttributeValue {
    def toValue: Seq[Any] = arr.map(_.toValue)
    def toJson = JsArray(arr.map(_.toJson).toVector)
  }

  case class MapAttrValue(map: Map[String, AttributeValue]) extends AttributeValue {
    def toValue: Map[String, Any] = map.mapValues(_.toValue)
    def toJson = JsObject(map.mapValues(_.toJson))
  }

  object AttributeValue {
    object JsonProtocol {
      implicit val attributeValuesJsonFormat = new JsonFormat[AttributeValue] {
        def write(obj: AttributeValue): JsValue = obj.toJson
        def read(json: JsValue): AttributeValue = json match {
          case JsString(x) => StringAttrValue(x)
          case JsNumber(x) => NumericAttrValue(x.toDouble)
          case JsBoolean(x) => BooleanAttrValue(x)
          case JsArray(x) => ArrayAttrValue(x.map(read))
          case JsObject(x) if x.keySet == Set("lat", "long") =>
            GeoAttrValue(x("lat").convertTo[Double], x("long").convertTo[Double])
          case JsObject(x) => MapAttrValue(x.mapValues(read))
        }
      }
    }
  }

  implicit def stringAttributeWrap(str: String): AttributeValue = StringAttrValue(str)
  implicit def numericAttributeWrap[T](num: T)(implicit n: Numeric[T]): AttributeValue = NumericAttrValue(n.toDouble(num))
  implicit def booleanAttributeWrap(bool: Boolean): AttributeValue = BooleanAttrValue(bool)
  implicit def arrayAttributeWrap[T](arr: Array[T])(implicit w: T => AttributeValue): AttributeValue = ArrayAttrValue(arr.map(w))
  implicit def arrayAttributeWrap[T](arr: Seq[T])(implicit w: T => AttributeValue): AttributeValue = ArrayAttrValue(arr.map(w))
  implicit def mapAttributeWrap[T](map: Map[String, T])(implicit w: T => AttributeValue): AttributeValue = MapAttrValue(map.mapValues(w))
  implicit def mapAttributeWrap(map: Map[String, AttributeValue]): AttributeValue = MapAttrValue(map)

  implicit def stringAttributeUnwrap(attr: StringAttrValue): String = attr.toValue
  implicit def numericAttributeUnwrap(attr: NumericAttrValue): Double = attr.toValue
  implicit def booleanAttributeUnwrap(attr: BooleanAttrValue): Boolean = attr.toValue
  implicit def arrayAttributeUnwrap(attr: ArrayAttrValue): Seq[Any] = attr.toValue
  implicit def mapAttributeUnwrap(attr: MapAttrValue): Map[String, Any] = attr.toValue

  object JsonProtocol {
    import eu.shiftforward.adstax.storage.UserAttributes.AttributeValue.JsonProtocol._
    implicit object UserAttributesRootJsonFormat extends RootJsonFormat[UserAttributes] {
      def write(ua: UserAttributes) = ua.attributes.toJson
      def read(value: JsValue) = UserAttributes(value.convertTo[Map[String, AttributeValue]])
    }
  }
}
