package eu.shiftforward.adstax.scheduler.action

import scala.concurrent.duration.{ Duration, FiniteDuration }

import eu.shiftforward.adstax.scheduler.RecurrentScript
import eu.shiftforward.adstax.scheduler.RecurrentScript.JsonProtocol._
import spray.json._
import spray.json.DefaultJsonProtocol._

import eu.shiftforward.adstax.scheduler.action.SchedulerAction.RepetitionConfig

/**
 * A `SchedulerAction` capable of publishing messages to a RabbitMQ broker. The message definition builder script
 * should return an object with fields `exchangeKey`, `routingKey` and `message`.
 *
 * @param id the id of the SchedulerAction
 * @param in the duration in which to run the action
 * @param repeat the duration in which to repeat the scheduled action, or `None` if it is a one-shot task
 * @param messageDefinitionBuilder the script capable of recurrently building messages.
 */
case class RabbitMQSchedulerAction(
    id: String,
    in: FiniteDuration,
    repeat: Option[RepetitionConfig],
    messageDefinitionBuilder: RecurrentScript) extends SchedulerAction {
  val action = RabbitMQSchedulerAction.Action
}

object RabbitMQSchedulerAction {

  final val Action = "publish-to-rabbit-mq"

  object JsonProtocol {
    /**
     * Enables conversions of `RabbitMQSchedulerAction` to and from JSON.
     */
    implicit object RabbitMQSchedulerActionJsonFormat extends RootJsonFormat[RabbitMQSchedulerAction] {

      def write(r: RabbitMQSchedulerAction) = {
        val fields = Map(
          "action" -> r.action.toJson,
          "id" -> r.id.toJson,
          "in" -> r.in.toMillis.toJson,
          "messageDefinition" -> r.messageDefinitionBuilder.toJson) ++
          (r.repeat.map { repeat => Map("repeat" -> repeat.toJson) } getOrElse { Map() })
        JsObject(fields)
      }

      def read(json: JsValue) = {
        val obj = json.asJsObject
        obj.getFields("id", "in", "messageDefinition", "missedExecutionBehavior") match {
          case Seq(JsString(id), JsNumber(in), messageDefinition: JsValue, behavior: JsString) =>
            RabbitMQSchedulerAction(
              id,
              Duration.fromNanos(in.toLong * 1000000l),
              obj.fields.get("repeat").collect { case repeat: JsObject => repeat.convertTo[RepetitionConfig] },
              messageDefinition.convertTo[RecurrentScript]
            )

          case _ =>
            deserializationError("One or more of the mandatory parameters for a RabbitMQ scheduler action" +
              "(id, in, messageDefinition) are missing or have an invalid type.")
        }
      }
    }
  }
}
