package eu.shiftforward.adstax.recommender

import spray.json.{ JsObject, JsValue, DefaultJsonProtocol, RootJsonFormat }
import eu.shiftforward.adstax.recommender.Features.JsonProtocol._

sealed trait RecommendationQuery {
  def uidToSegment: Option[String]
  def clientId: Option[String]
  def siteId: Option[String]
  def getId: String
  def getNonPersonalizedId = Recommendation.getId(None, None)
  def features: Option[Features]
}

/**
 * Represents a query to retrieve a recommendation for a user id.
 *
 * @param userId the user Id
 * @param clientId the client Id or None
 * @param siteId the siteId or None
 * @param features the features related to this query
 */
case class UserRecommendationQuery(
    userId: String,
    override val clientId: Option[String],
    override val siteId: Option[String],
    override val features: Option[Features] = None) extends RecommendationQuery {
  def uidToSegment = Some(userId)
  def getId = (Some(userId), None).toString()
}

/**
 * Represents a query to retrieve a recommendation for a product id.
 *
 * @param productId the product Id
 * @param clientId the client Id or None
 * @param siteId the site Id or None
 * @param uidToSegment the user Id to be used for segmentation
 * @param features the features related to this query
 */
case class ProductRecommendationQuery(
    productId: String,
    override val clientId: Option[String],
    override val siteId: Option[String],
    override val uidToSegment: Option[String] = None,
    override val features: Option[Features] = None) extends RecommendationQuery {
  def getId = (None, Some(productId)).toString()
}

object RecommendationQuery {
  object JsonProtocol extends DefaultJsonProtocol {

    implicit val userRecommendationQueryFormat = jsonFormat4(UserRecommendationQuery)
    implicit val productRecommendationQueryFormat = jsonFormat5(ProductRecommendationQuery)

    implicit object RecommendationQueryFormat extends RootJsonFormat[RecommendationQuery] {
      override def write(obj: RecommendationQuery): JsValue = obj match {
        case urq: UserRecommendationQuery => userRecommendationQueryFormat.write(urq)
        case prq: ProductRecommendationQuery => productRecommendationQueryFormat.write(prq)
      }

      override def read(json: JsValue): RecommendationQuery = {
        val JsObject(map) = json.asJsObject
        if (map.contains("userId")) userRecommendationQueryFormat.read(json)
        else if (map.contains("productId")) productRecommendationQueryFormat.read(json)
        else throw new NotImplementedError()
      }
    }
  }
}
