package eu.shiftforward.adstax.recommender

import com.github.nscala_time.time.Imports._
import eu.shiftforward.adstax.util.RmqRpcJsonTypeRoutingKey
import spray.json._

/**
 * Represents a supported operation of the RecommenderActor.
 */
sealed trait RecommenderOperation {
  /**
   * The unique operation type for a concrete RecommenderOperation.
   */
  def operation: String = this.getClass.getSimpleName
}

/**
 * Gets a recommendation.
 * @param query the query for the recommendation (userId, siteId, ...)
 */
case class GetRecommendation(query: RecommendationQuery) extends RecommenderOperation

object GetRecommendationTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[GetRecommendation] {
  override def value: String = "getrecommendation.json"
}

/**
 * Updates recommendations in bulk, for a given clientId and siteId, if provided.
 * @param dateTime the timestamp for which to update recommendations. This should enforce that only events prior to the
 *                 provided timestamp are considered for the new recommendations
 * @param clientId the optional clientId, in order to filter events
 * @param siteId the optional siteId, in order to filter events
 */
case class UpdateRecommendations(dateTime: DateTime, clientId: Option[String] = None, siteId: Option[String] = None)
  extends RecommenderOperation

object UpdateRecommendationsTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[UpdateRecommendations] {
  override def value: String = "updaterecommendations.json"
}

object RecommenderOperation {
  object JsonProtocol extends DefaultJsonProtocol {
    import RecommendationQuery.JsonProtocol._

    implicit object GetRecommendationJsonFormat extends RootJsonFormat[GetRecommendation] {
      def write(getRecommendation: GetRecommendation): JsValue = JsObject(Map(
        "operation" -> JsString(getRecommendation.operation),
        "query" -> getRecommendation.query.toJson))

      override def read(json: JsValue): GetRecommendation =
        GetRecommendation(json.asJsObject.fields.get("query").get.convertTo[RecommendationQuery])
    }

    implicit object UpdateRecommendationsJsonFormat extends RootJsonFormat[UpdateRecommendations] {
      def write(updateRecommendations: UpdateRecommendations): JsValue = {
        val fields =
          Map(
            "operation" -> JsString(updateRecommendations.operation),
            "dateTime" -> JsString(updateRecommendations.dateTime.toString)) ++
            (updateRecommendations.clientId.map { cId => Map("clientId" -> JsString(cId)) } getOrElse { Map() }) ++
            (updateRecommendations.siteId.map { sId => Map("siteId" -> JsString(sId)) } getOrElse { Map() })

        JsObject(fields)
      }

      def read(json: JsValue): UpdateRecommendations = {
        val obj = json.asJsObject.fields
        obj.get("dateTime") match {
          case Some(JsString(dateTime)) =>
            UpdateRecommendations(
              new DateTime(dateTime),
              obj.get("clientId").collect { case JsString(cId) => cId },
              obj.get("siteId").collect { case JsString(sId) => sId })

          case _ =>
            throw new DeserializationException("The mandatory parameter 'dateTime' is missing or has an invalid type.")
        }
      }
    }
  }
}
