package eu.shiftforward.adstax.util

import akka.actor.{ Actor, ActorRef, Stash }
import com.github.sstone.amqp.ChannelOwner.{ Connected, Disconnected }

import scala.concurrent.Future
import scala.util.{ Failure, Success }

/**
 * A trait that extends a rmq publisher and implements a common pattern where all received messages
 * are stashed until the underlying producer is created and connected.
 *
 * Mixing in this class requires defining the `producerConnected` method that should contain all the
 * logic of publishing to rmq and can assume all is ready for publishing, as well as defining the
 * `setupRmq` method which should create the producer actor. Another aspect of using
 * this trait is that instead of using "context.become(newReceive)", one should use
 * "stashedContextBecome(newReceive)" so that producer disconnected handling behaviour is not lost
 * while changing context.
 */
trait BaseRmqProducerStashActor extends Actor with Stash {
  import BaseRmqProducerStashActor._
  import context.dispatcher

  /**
   * Setup rmq producer initialization and optionally exchange declaration
   * @return a future of the producer actorRef, not necessarily fully connected to rmq
   */
  def setupRmq: Future[ActorRef]

  override def preStart() {
    self ! ProducerReady
    setupRmq.onComplete {
      case Success(producer) => self ! ProducerCreated(producer)
      case Failure(throwable) => throw throwable
    }
  }

  /**
   * Changes actor context assuming all required pre-conditions are met
   *
   * @param producer the rmq producer actor
   * @param interrupted if true, a previous context is stored, otherwise changes to new
   *                    producerConnected receive
   */
  private def becomeConnected(producer: ActorRef, interrupted: Boolean) = {
    unstashAll()
    if (interrupted) context.unbecome()
    else stashedContextBecome(producer)(producerConnected(producer))
  }

  /**
   * Underlying producer actor is not ready state
   *
   * @param producer the optional producer actor if it has already been created
   * @param connected whether the producer actor is connected
   * @param interrupted whether connection was established and lost afterwords.
   *                    Required to return to previous context after reconnection
   * @return the producer not ready receive
   */
  private def producerNotConnected(
    producer: Option[ActorRef],
    connected: Boolean,
    interrupted: Boolean = false): Receive = {

    case ProducerCreated(prod) =>
      if (connected) {
        becomeConnected(prod, interrupted)
      } else {
        context.become(producerNotConnected(Some(prod), connected, interrupted))
      }

    case Connected =>
      if (producer.isDefined) {
        becomeConnected(producer.get, interrupted)
      } else {
        context.become(producerNotConnected(producer, connected = true, interrupted))
      }

    case other =>
      stash()
  }

  /**
   * Handles rmq producer disconnection by starting over the state machine and awaiting for a
   * connection message again. It stores the previous state to change back after reconnection.
   * @param producer the rmq producer actor
   * @return the producerNotConnected state
   */
  private def producerDisconnected(producer: ActorRef): Receive = {
    case Disconnected =>
      context.become(producerNotConnected(
        producer = Some(producer),
        connected = false,
        interrupted = true), discardOld = false)
  }

  /**
   * Method for changing actor context while keeping necessary handling control behaviour.
   *
   * @param producer the rmq producer actor
   * @param newContext the new context to change
   */
  def stashedContextBecome(producer: ActorRef)(newContext: Receive): Unit = {
    context.become(producerDisconnected(producer) orElse newContext)
  }

  /**
   * Logic for handling rmq publishing with assurance that everything is correctly created and
   * connected.
   *
   * @param producer the rmq producer actor
   * @return the logic handling receive
   */
  def producerConnected(producer: ActorRef): Receive

  final def receive = producerNotConnected(None, connected = false)
}

object BaseRmqProducerStashActor {
  private[BaseRmqProducerStashActor] case class ProducerCreated(producer: ActorRef)

  /**
   * Message that when received indicates the producer is fully operational.
   * Most of the time it can be ignored and all will function as expected. May be required for
   * custom implementations that need to be aware of this.
   */
  case object ProducerReady
}
