package eu.shiftforward.adstax.scheduler.rpc

import scala.concurrent.{ ExecutionContext, Future }

import akka.actor.ActorRefFactory
import com.typesafe.config.Config

import eu.shiftforward.adstax.scheduler.SchedulerOperationResult.JsonProtocol._
import eu.shiftforward.adstax.scheduler._
import eu.shiftforward.adstax.scheduler.action.SchedulerAction
import eu.shiftforward.adstax.scheduler.rpc.SchedulerOperation.JsonProtocol._
import eu.shiftforward.adstax.util.{ RmqRpcJsonClient, RmqRpcJsonClientTypeDescriptor }

/**
 * The AMQP RPC Client to interact with the scheduler.
 */
trait SchedulerRmqRpcClient extends RmqRpcJsonClient with SchedulerClient {
  def actorRefFactory: ActorRefFactory
  def rmqActorRefFactory = actorRefFactory

  implicit val scheduleTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[Schedule, Scheduled] {}

  implicit val cancelTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[Cancel, Cancelled] {}

  implicit val getStatusTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[GetStatus, JobsStatus] {}

  implicit val getJobStatusTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[GetJobStatus, Option[JobStatus]] {}

  implicit val scheduleRequestRoutingKey = ScheduleTypeRoutingKey
  implicit val cancelRequestRoutingKey = CancelTypeRoutingKey
  implicit val getStatusRequestRoutingKey = GetStatusTypeRoutingKey
  implicit val getJobStatusRequestRoutingKey = GetJobStatusTypeRoutingKey

  def scheduleAction(action: SchedulerAction): Future[SchedulerAction] =
    dispatchRequest[Schedule, Scheduled] {
      Schedule(action)
    }.map(_.action)

  def cancelAction(id: String): Future[String] =
    dispatchRequest[Cancel, Cancelled] {
      Cancel(id)
    }.map(_.actionId)

  def getJobsStatus: Future[Map[String, JobStatus]] =
    dispatchRequest[GetStatus, JobsStatus] {
      GetStatus
    }.map(_.jobs)

  def getJobStatus(id: String): Future[Option[JobStatus]] =
    dispatchRequest[GetJobStatus, Option[JobStatus]] {
      GetJobStatus(id)
    }
}

object SchedulerRmqRpcClientFactory {
  def apply(_actorRefFactory: ActorRefFactory, _ec: ExecutionContext, _rmqConfig: Config,
    _rpcClientConfig: Config): SchedulerRmqRpcClient = {
    new SchedulerRmqRpcClient {
      def actorRefFactory = _actorRefFactory
      implicit def ec: ExecutionContext = _ec
      def rmqConfig = _rmqConfig
      def rpcClientConfig = _rpcClientConfig
    }
  }
}
